require 'json'
require 'pe_backup_tools/utils'

module PeBackupTools
  module Utils
    # This module provides helper functions for node classification
    module Classifier
      def self.get_classification(certname, classifier_host, cert_conf, facts = {}, trusted_facts = {})
        request = {}
        request['fact'] = facts if !facts.nil? && !facts.keys.empty?
        request['trusted'] = trusted_facts if !trusted_facts.nil? && !trusted_facts.keys.empty?
        classifier_url = "https://#{classifier_host}:4433/classifier-api/v1/classified/nodes/#{certname}"
        request_body = JSON.generate(request)
        response = PeBackupTools::Utils.post_response(classifier_url, cert_conf, request_body)
        JSON.parse(response.body)
      end

      def self.unpin_from_all(nodes, classifier_host, cert_conf)
        classifier_url = "https://#{classifier_host}:4433/classifier-api/v1/commands/unpin-from-all"
        request_body = JSON.generate(nodes: nodes)
        response = PeBackupTools::Utils.post_response(classifier_url, cert_conf, request_body)
        JSON.parse(response.body)
      end

      def self.pin_nodes_to_group(id, nodes, classifier_host, cert_conf)
        classifier_url = "https://#{classifier_host}:4433/classifier-api/v1/groups/#{id}/pin"
        request_body = JSON.generate(nodes: nodes)
        PeBackupTools::Utils.post_response(classifier_url, cert_conf, request_body)
      end

      def self.get_node_group(node_group_name, classifier_host, cert_conf)
        nodes = get_list_of_node_groups(classifier_host, cert_conf)
        nodes.each do |node|
          return node if node['name'] == node_group_name
        end
        nil # return nil if no matching name found
      end

      def self.get_list_of_node_groups(classifier_host, cert_conf)
        classifier_url = "https://#{classifier_host}:4433/classifier-api/v1/groups"
        response = PeBackupTools::Utils.get_response(classifier_url, cert_conf)
        JSON.parse(response.body)
      end

      def self.set_node_group_param(node_group_name, param, class_name, value, classifier_host, cert_conf)
        node_group = get_node_group(node_group_name, classifier_host, cert_conf)
        if node_group && node_group['classes'][class_name][param]
          node_group['classes'][class_name][param] = value
          update_node_group(node_group, classifier_host, cert_conf)
        end
      end

      # Update string values in parameter arrays, or a string parameter.
      # Replace old_value with new_value and remove values that match items in the remove_list
      # To only replace items, call this method with remove_list set to nil
      # To only remove items, call this method with old_value set to nil
      def self.update_node_group_params(node_group_names, parameters, class_name, old_value, new_value, remove_list, classifier_host, cert_conf)
        node_group_names.each do |node_group_name|
          node_group = get_node_group(node_group_name, classifier_host, cert_conf)
          next unless node_group
          parameters.each do |param|
            next unless node_group['classes'][class_name][param]
            current_value = node_group['classes'][class_name][param]
            case current_value
            when String
              node_group['classes'][class_name][param].gsub!(old_value, new_value) if old_value
              node_group['classes'][class_name][param] = nil if remove_list && remove_list.include?(current_value)
            when Array
              node_group['classes'][class_name][param].map { |n| n.gsub!(old_value, new_value) } if old_value
              node_group['classes'][class_name][param].delete_if { |n| remove_list.any? { |v| n.match?(/#{v}/) } } if remove_list
            else
              raise "Invalid parameter type found for #{param}: #{param.class}"
            end
          end
          update_node_group(node_group, classifier_host, cert_conf)
        end
      end

      def self.update_node_group(node_group, classifier_host, cert_conf)
        classifier_url = "https://#{classifier_host}:4433/classifier-api/v1/groups/#{node_group['id']}"
        PeBackupTools::Utils.post_response(classifier_url, cert_conf, node_group.to_json)
      end
    end
  end
end
