require 'open3'
require 'puppet'
require 'puppet/util/execution'

module PeBackupTools
  module Utils
    # Commonly used utility functions used across the whole app.
    # These are mixins to save time and readability not having
    # to use the full namespace.
    module Mixins
      def error(message, command = nil, output = nil)
        # Message should be i18n before being passed to this function
        # e.g. error(_("This is my error message), command, output)
        text = message
        text += _("\nCommand: %{command}") % { command: command } if command
        text += _("\nOutput: %{output}") % { output: output } if output
        Puppet.err(text)
        exit 1
      end

      def warning(message, command = nil, output = nil)
        # Message should be i18n before being passed to this function
        # e.g. error(_("This is my error message), command, output)
        text = message
        text += _("\nCommand: %{command}") % { command: command } if command
        text += _("\nOutput: %{output}") % { output: output } if output
        Puppet.warning(text)
      end

      # Because we want the ability to return separate stdout and stderr,
      # we use Open3 instead of Puppet::Util::Execute, which will either
      # return a combined stdout/stderr like 2e, or it will throw away
      # stderr completely.
      #
      # Also, if we are not logging at the debug level, this bypasses
      # Puppet's logging mechanism to log directly into the log file
      # so we can record the output of these commands there.
      def run_command(command, combine: true)
        Puppet.debug(_('Running %{command}') % { command: command })
        if combine
          output, status = Open3.capture2e(command)
          Puppet.debug(_("Exited %{status} with the following output:\n%{output}") % { status: status.exitstatus, output: output })
          write_to_logfile("Command: #{command}\nOutput: #{output}\n") unless Puppet::Util::Log.level == :debug
          return output, status
        else
          stdout, stderr, status = Open3.capture3(command)
          Puppet.debug(_("Exited %{status} with the following output:\nStdout:\n%{stdout}\nStderr:\n%{stderr}") % { status: status.exitstatus, stdout: stdout, stderr: stderr })
          write_to_logfile("Command: #{command}\nStdout: #{stdout}\nStderr: #{stderr}\n") unless Puppet::Util::Log.level == :debug
          return stdout, stderr, status
        end
      end

      # This detects if a log file has been set up. If so, write directly
      # into it. If not, do nothing.
      def write_to_logfile(message)
        log = Puppet::Util::Log.destinations.select { |k, _v| k.is_a?(String) && k.include?('.log') }
        return if log.empty?
        log = log.keys.first
        File.write(log, message, mode: 'a')
      end

      def with_agent_disabled(message)
        Puppet.info(_('Disabling puppet agent'))
        command = "/opt/puppetlabs/bin/puppet agent --disable '#{message}'"
        output, status = run_command(command)
        unless status.exitstatus.zero?
          warning(_('Could not disable puppet agent. The action will still likely work, but ensure the agent is enabled after it finishes.'), command, output)
        end
        yield
      ensure
        Puppet.info(_('Re-enabling puppet agent'))
        command = '/opt/puppetlabs/bin/puppet agent --enable'
        output, status = run_command(command)
        unless status.exitstatus.zero?
          warning(_('Could not enable puppet agent. Ensure the agent is enabled after this action finishes.'), command, output)
        end
      end

      def wait_for_agent_lock
        command = '/opt/puppetlabs/bin/puppet config print agent_catalog_run_lockfile'
        output, status = run_command(command)
        unless status.exitstatus.zero?
          warning('Error reading the agent lock file setting. Unable to verify that the agent is not currently running.', command, output)
          return
        end
        lockfile = output.strip
        waits = 0
        while File.exist?(lockfile) && waits < 300
          Puppet.info(_("Agent run with PID #{File.read(lockfile).strip} currently in progress. Waiting up to 5 minutes for the run to finish. Wait time elapsed: #{waits} seconds.")) if (waits % 30).zero?
          waits += 1
          sleep 1
        end
        if File.exist?(lockfile)
          error(_('The agent run in progress did not finish within 5 minutes. Wait for this agent run to finish, then run the command again.'))
        end
      end
    end
  end
end
