#
# @summary  A Puppet plan is to setup the agent package for a given platform on PE server
#
# @param os [String]
#   The name of the OS for which the agent to be added to the PE server
#
# @param os_version [String]
#   The version of the OS to be added to the PE server
#
# @param architecture [String]
#   The architecture of the platform to be added to the PE server
#   The architecture is required only for Windows platform
#
# @params fips [Boolean]
#   A boolean value to enable FIPS on the platform
#
# @param target [TargetSpec]
#   The name of target (PE primary and any replicas that have been provisioned) where the plan will get executed.
#
# @param codename [Optional[String]]
#   The codename of the platform to be added to the PE server
#   The codename is required only for debian and ubuntu platforms
#
# @param agent_version [Optional[String]]
#   The puppet agent version for the platform to be supported.
#
#
plan enterprise_tasks::add_platform (
  String[1]  $os,
  String[1]  $os_version,
  String[1]  $architecture,
  Boolean    $fips,
  TargetSpec $target,
  Optional[String[1]] $codename      = undef,
  Optional[String[1]] $agent_version = undef,
) {
  $os_formatted = $os.strip.downcase
  $os_version_formatted = $os_version.strip.downcase
  $architecture_formatted = $architecture.strip.downcase
  $code_name = $codename ? {
    undef   => $codename,
    default => $codename.strip.downcase,
  }

  $platform_name = enterprise_tasks::platform_name($os_formatted, $architecture_formatted, $os_version_formatted, $fips)
  if $platform_name =~ /^debian|^ubuntu|^osx/ and $codename == undef {
    fail('Codename is required for Debian/OSX/Ubuntu')
  }

  $platform = enterprise_tasks::platform($platform_name)
  if $platform == undef {
    fail("Platform ${platform_name} is not supported")
  }

  # $pe_version parameter conflicts with the point of this plan, 
  # therefore it is set to `undef`
  $pe_version = undef
  $resource_options = enterprise_tasks::resource_options(
    $platform_name,
    $os_formatted,
    $architecture_formatted,
    $os_version_formatted,
    $fips,
    $code_name,
    $agent_version,
    $pe_version
  )

  apply_prep($target)
  $result = apply($target, '_description' => 'Running manifest to add new platform on PE', '_catch_errors' => true) {
    include puppet_enterprise::packages
    class { 'pe_repo' : }
    create_resources("pe_repo::${platform}", $resource_options)
  }
  if !$result[0].ok {
    fail("Platform ${platform_name} package failed to download. Check the supported platform or architecture")
  }
}
