# Installs several bootstrap files that are not normally managed in the Puppet
# Enterprise catalog.
#
# These files were installed by the puppet-enterprise-installer shim previously.
#
# * /etc/puppetlabs/puppet/hiera.yaml
# * /opt/puppetlabs/bin/puppet-enterprise-uninstaller
# * /opt/puppetlabs/server/pe_build
# * /opt/puppetlabs/server/effective_date
#
# (See {pe_install::bootstrap} for parameter docs.)
class pe_install::bootstrap::install_tarball_files(
  Pe_install::Absolute_path $pe_tarball_dir,
  Pe_install::Effective_date $effective_date,
  Pe_install::Pe_build $pe_build_version,
  Boolean $manage_permissions = false,
) {
  include 'pe_install::params'

  $file_opts = $manage_permissions ? {
    true    => {
      owner => 'root',
      group => 'root',
      mode  => '0644',
    },
    default => {}
  }

  $etc_dirs = [
    '/etc/puppetlabs',
    $pe_install::params::puppet_etc_dir,
  ]
  file { $etc_dirs:
    ensure => 'directory',
    *      => $file_opts,
  }

  # Install the default hiera.yaml with Classifier data hierarchy only on
  # initial install.  Do not replace, user's may customize this file.
  #
  # NOTE: This needs to be installed before puppet-agent, as the agent will install
  # a more basic hiera.yaml default without the Classifier data hierarchy.
  # (This was originally in the shim:
  # https://github.com/puppetlabs/pe-installer-shim/commit/6a4ca2a
  file { "${pe_install::params::puppet_etc_dir}/hiera.yaml":
    ensure  => 'file',
    source  => 'puppet:///modules/pe_install/hiera.yaml',
    replace => false,
    *       => $file_opts,
  }

  file { "${pe_install::params::server_dir}/effective_date":
    ensure  => 'file',
    content => $effective_date,
    *       => $file_opts,
  }

  file { "${pe_install::params::server_dir}/pe_build":
    ensure  => 'file',
    content => $pe_build_version,
    *       => $file_opts,
  }

  # TODO: Any reason to keep this file in the tarball?
  # A `meep uninstall --targets` could run it for you...
  file { $pe_install::params::puppetlabs_bin_dir:
    ensure => 'directory',
    *      => $file_opts,
  }
  file { "${pe_install::params::puppetlabs_bin_dir}/puppet-enterprise-uninstaller":
    ensure => 'file',
    source => "${pe_tarball_dir}/puppet-enterprise-uninstaller",
    *      => $file_opts + { mode => '0755' },
  }
}
