# This is a stand alone class to rollback a failed migration from one major
# postgresql version to another.
#
# It completely removes the data for the given $new_postgres_version
# of PostgreSQL and resets package and service state such that the
# ./puppet-enterprise-installer can once more migrate PostgreSQL to
# $new_postgres_version.
#
# This should only ever be used if the pg_upgrade fails during the initial
# upgrade of PE. And this assumes that whatever underlying problem caused
# pg_ugprade to fail has been resolved.
#
# @param new_postgres_version [String] The version of PostgreSQL we are upgrading TO.
# @param delete_datastore [Boolean] Actually delete the datastore. Defaults to false. Must
#   be explicitly set true before the class will do anything other than warn
#   about what it intends to do and fail. 
class pe_install::upgrade::rollback_postgres_migration(
  String $new_postgres_version,
  Boolean $delete_datastore = false,
) {
  include puppet_enterprise::params

  $new_package = "pe-postgresql${new_postgres_version}"
  $pg_data_dir = '/opt/puppetlabs/server/data/postgresql'
  $new_data_dir = "${pg_data_dir}/${new_postgres_version}/data"
  $new_tablespaces = "${pg_data_dir}/{activity,classifier,inventory,orchestrator,puppetdb,rbac}/PG_${new_postgres_version}*"

  warning("This class will remove the ${new_data_dir} and ${new_tablespaces} directories.")
  warning("The Postgresql ${new_postgres_version} datastore will be completely deleted.")
  warning("And it will remove all of the ${new_package} packages from the system.")

  if $new_postgres_version != $puppet_enterprise::params::postgres_version {
    warning("The 'new_pe_version' parameter has been set to '${new_postgres_version}'.")
    warning("However puppet_enterprise::params::postgres_version indicates'${puppet_enterprise::params::postgres_version}'.")
    warning('Is the wrong version of pe-modules installed?')
    fail('Unexpected postgres version mismatch.')
  } elsif $delete_datastore {
    # Delete the database!
    package { ["${new_package}-pglogical","${new_package}-pgrepack"]:
      ensure =>  absent,
    }
    -> package { ["${new_package}-contrib","${new_package}-server"]:
      ensure  => absent,
    }
    -> package { $new_package:
      ensure => absent,
    }
    -> exec { "Remove ${new_package} datadir at ${new_data_dir}":
      command => "rm -rf ${new_data_dir}",
      path    => '/bin:/sbin:/usr/sbin',
    }
    -> exec { "Remove ${new_package} tablespaces at ${new_tablespaces}":
      command => "rm -rf ${new_tablespaces}",
      path    => '/bin:/sbin:/usr/sbin',
    }
    -> service { 'pe-postgresql':
      ensure => running
    }
    notice('This should allow puppet-enterprise-installer to attempt to upgrade again.')
  } else {
    fail("Please set the 'delete_datastore' parameter to true if you wish to proceed.")
  }
}
