
# == Class: puppet_enterprise::certs::puppetdb_allowlist
#
# A class for managing the puppetdb certificate allowlist. In addition to
# explicitly specified certnames, this does a query to puppetdb to add any nodes
# which are classified as Puppetdb, Master, Console, or Orchestrator.
#
# A certificate allowlist is a basic text file with one certificate
# per line that an application will load and parse.
#
# === Parameters
#
# [*certnames*]
#   Array of Strings. The names of the certificates which should be always be
#   added to the allowlist, in addition those from the dynamic puppetdb query.
#
# === Examples
#
#   puppet_enterprise::certs::puppetdb_allowlist_entry { "${certificate_allowlist_target}":
#     allowlisted_certnames => ['mom.infra.net', 'console.infra.net']
#   }
#
class puppet_enterprise::certs::puppetdb_allowlist(
  String $cert_allowlist_path = '/etc/puppetlabs/puppetdb/certificate-allowlist',
  Array $certnames = [],
) {
  include puppet_enterprise::params

  # "storeconfigs" being true is used here to determine if PuppetDB is ready
  # to accept queries. This only matters during a PE installation when
  # templates are applied. This setting is typically false then, since a
  # manifest might otherwise attempt to query PuppetDB before it was running.
  #
  # Replicas are removed from this list as they should be covered by the
  # sync_allowlist parameter in the puppetdb profile, and there are times
  # when we want to unsync PuppetDB on the replica for upgrade purposes.
  if $settings::storeconfigs {
    $allowlist_query_result =
      puppetdb_query(
                    ['from', 'resources',
                      ['extract', 'certname',
                        ['and',
                          ['and',
                            ['=', 'type', 'Class'],
                            ['=', ['node','active'], true],
                            ['or',
                              ['=', 'title', 'Puppet_enterprise::Profile::Puppetdb'],
                              ['=', 'title', 'Puppet_enterprise::Profile::Master'],
                              ['=', 'title', 'Puppet_enterprise::Profile::Orchestrator'],
                            ]
                          ],
                          ['not',
                            ['in', 'certname',
                              ['from', 'resources',
                                ['extract','certname',
                                  ['and',
                                    ['=', 'type', 'Class'],
                                    ['=', ['node', 'active'], true],
                                    ['=', 'title', 'Puppet_enterprise::Profile::Primary_master_replica']
                                  ]
                                ]
                              ]
                            ]
                          ]
                        ]
                      ]
                    ]
                    )

    $default_allowlist_certnames = $allowlist_query_result.map |$row| { $row['certname'] }
  }
  else {
    $default_allowlist_certnames = []
  }

  pe_union($default_allowlist_certnames, $certnames).each |$cn| {
    puppet_enterprise::certs::allowlist_entry { "${title} entry: ${cn}":
      certname => $cn,
      target   => $cert_allowlist_path,
      require  => File[$cert_allowlist_path],
    }
  }
}
