# This class configures services which manage puppet code deployment in PE infrastructure
#
# @param certname [String] The name of the the node's SSL certificate.
# @param puppetserver_codedir [String] Location where puppetserver reads code from
# @param puppet_default_codedir [String] Default location where puppet reads code from
# @param versioned_deploys_codedir [String] Location where puppetserver reads code from when 'versioned_deploys' is enabled
#                                           (i.e. versioned_deploys_enabled is true)
# @param code_manager_auto_configure [Boolean] Configure code-manager based on r10k params
# @param code_manager_certname [String] The name of the code manager's SSL certificate.
# @param versioned_deploys_enabled [Boolean] Set to 'true' to enable the versioned deploys feature of puppetserver/code manager
# @param ssl_listen_port [Integer] What port puppetserver is listening on
# @param primary_certname [String] The certname of the PE primary
# @param replication_mode ['source', 'replica', 'none'] Which kind of replica this server is, if any.
# @param provisioned_replicas [Array[String]] Array of certnames for provisioned replicas; these will be given access to the ca-data file
#                                             sync repo.
# @param cert_data_discard_history [Boolean] Discard old cert data
# @param environmentpath [String] Set environmentpath in puppet.conf
# @param r10k_known_hosts Array[Hash] The host names, types, and public keys that r10k should trust.
#                                     The format in the console should be [{"name": "github.com",
#                                                                           "type": "ecdsa-sha2-nistp256",
#                                                                           "key": "AAAAE2V...."}, ...]
# @param r10k_remote [String] The git url for the r10k configuration
# @param r10k_proxy [String] Sets an HTTP proxy for r10k to use for all HTTP/HTTPS operations performed by r10k, such as
#                            requests to the forge
# @param r10k_postrun [String] An optional command that will be run after r10k finishes deploying environments. The command must be an
#                              array of strings that will be used as an argument vector.
# @param code_id_command [String] The absolute path to a script to run to generate the code_id for an environment.
# @param code_content_command [String] The absolute path to a script to run to retrieve a file from an environment for a given code_id.
# @param file_sync_enabled [Boolean] Flag to enable file sync, leave undef if using code_manager_auto_configure
# @param file_sync_versioned_sync_pool [Integer] The number of threads used deploy environments in parallel when using versioned dirs.
#                                                defaults to 2.
# @param file_sync_copy_method [Enum['java','shell-cp'] The underlying copy implementation to use in versioned deploys. The 'java'
#                                                       option will use Files.copy, while 'shell-cp' will use `/bin/cp`. defaults to
#                                                      'shell-cp'.
# @param git_provider [String] The git provider to use for r10k configuration, either 'rugged' or 'shellgit'
# @param git_private_key [String] The rugged private key path for pe_r10k configuration
# @param git_default_ref Optional[String] The default git ref to use when resolving modules, default undef
# @param git_oauth_token Optional[String] The OAuth token to use for git servers, default undef
# @param git_proxy Optional[String] The proxy all git operations will use, default undef
# @param git_repositories Optional[Array[Hash]] A list of repositories with special settings
# @param forge_proxy [String] The proxy all Puppet Forge operations will use
# @param forge_baseurl [String] the Puppet Forge API Base URL (e.g. "https://forgeapi.puppet.com")
# @param forge_authorization_token [String] Puppet Forge Authorization Token
class puppet_enterprise::master::code_management (
  String $certname,
  String $puppetserver_codedir,
  String $puppet_default_codedir,
  String $versioned_deploys_codedir,
  Boolean $code_manager_auto_configure,
  String $code_manager_certname,
  Boolean $versioned_deploys_enabled,
  Integer $ssl_listen_port,
  String $primary_certname,
  Puppet_enterprise::Replication_mode $replication_mode,
  Array[String] $provisioned_replicas,
  Optional[Array[Hash]] $git_repositories                   = [],
  Optional[Boolean] $cert_data_discard_history              = true,
  Optional[String] $environmentpath                         = undef,
  Optional[Array[Hash]] $r10k_known_hosts                   = [],
  Optional[String] $r10k_remote                             = undef,
  Optional[String] $r10k_proxy                              = undef,
  Optional[Array]  $r10k_postrun                            = undef,
  Optional[String] $code_id_command                         = undef,
  Optional[String] $code_content_command                    = undef,
  Optional[Enum['rugged', 'shellgit']] $git_provider        = undef,
  Optional[String] $git_private_key                         = undef,
  Optional[String] $git_default_ref                         = undef,
  Optional[String] $git_oauth_token                         = undef,
  Optional[String] $git_proxy                               = undef,
  Optional[Boolean] $file_sync_enabled                      = undef,
  Optional[Enum['shell-cp', 'java']] $file_sync_copy_method = 'shell-cp',
  Optional[Integer] $file_sync_versioned_sync_pool          = 2,
  Optional[String] $forge_proxy                             = undef,
  Optional[String] $forge_baseurl                           = undef,
  Optional[String] $forge_authorization_token               = undef,
) {
  $is_replica = ($replication_mode == 'replica')
  $is_compiler = pe_compile_master($replication_mode)
  $is_primary = !($is_replica or $is_compiler)

  [$code_manager_git_config, $r10k_git_config] = puppet_enterprise::parse_code_mgmt_git_config(
    $git_provider,
    $git_private_key,
    $git_default_ref,
    $git_oauth_token,
    $git_proxy,
    $git_repositories
  )

  [$code_manager_forge_config, $r10k_forge_config] = puppet_enterprise::parse_code_mgmt_forge_config(
    $forge_proxy,
    $forge_baseurl,
    $forge_authorization_token,
  )

  # The code manager class only applies to the Primary, but
  # deploy settings should be set everywhere.
  if $code_manager_auto_configure {
    if ($file_sync_enabled == false) {
      fail('Configuring $code_manager_auto_configure requires $file_sync_enabled to be either true or undef. Please unset $file_sync_enabled before continuing.')
    }
    $deploy_settings = {
      'write_lock' => 'Direct invocation of r10k is disabled when Code Manager is auto configured.',
    }
    if $is_primary {
      # Auto configure code manager with the provided settings
      class {'puppet_enterprise::master::code_manager':
        certname                      => $code_manager_certname,
        remote                        => $r10k_remote,
        proxy                         => $r10k_proxy,
        git_provider                  => $git_provider,
        git_settings_from_code_mgmt   => $code_manager_git_config,
        forge_settings_from_code_mgmt => $code_manager_forge_config,
        puppet_master_port            => $ssl_listen_port,
        require                       => Package['pe-puppetserver'],
      }
    }
  } else {
    $deploy_settings = undef
  }

  # R10K config applies to all infra nodes.
  #
  # Technically we probably shoudn't be adding dormant r10k configuration on
  # compilers. However there may be users relying on r10k on their compilers if
  # they aren't using code manager but have their own code management setup. Also,
  # there may be someone out there relying on r10k existing on compilers because
  # we never explicitly told them not to.
  #
  # For now we allow this configuration to happen on compilers
  if ($r10k_remote and !pe_empty($r10k_remote)) or $code_manager_auto_configure {
    # If users declare r10k via the installer we automatically enable
    # rugged support if they specify a private_key so we have no
    # external dependencies on the os providing git. This also
    # allows us to hardcode a path to the key without knowing what
    # user account they will run r10k as in the future.

    if $git_private_key and !pe_empty($git_private_key) {
      pe_validate_absolute_path($git_private_key)

      # Because of the way set_owner_group_permissions works, we need to
      # set the same mode for contents of /etc/puppetlabs/puppetserver/ssh
      # and $git_private_key, otherwise, in the case of $git_private_key
      # being in /etc/puppetlabs/puppetserver/ssh, these two
      # set_owner_group_permissions will fight one another in every puppet run.
      $ssh_private_key_mode = '0600'
      puppet_enterprise::set_owner_group_permissions { '/etc/puppetlabs/puppetserver/ssh':
        file_mode   => $ssh_private_key_mode,
        owner       => 'pe-puppet',
        group       => 'pe-puppet',
        target_type => 'directory',
        dir_mode    => '0700',
        require     => Package['pe-puppetserver'],
      }

      if $is_replica {
        $_git_private_key_content = file($git_private_key, '/dev/null')
        if pe_empty($_git_private_key_content) {
          warning("The git_private_key at ${git_private_key} not found or empty on ${servername}; unable to sync to replica.")
        } else {
          file { $git_private_key:
            ensure  => file,
            content => Sensitive($_git_private_key_content),
            mode    => $ssh_private_key_mode,
            owner   => 'pe-puppet',
            group   => 'pe-puppet',
            require => Package['pe-puppetserver'],
          }
        }
      } else {
        puppet_enterprise::set_owner_group_permissions { $git_private_key:
          file_mode   => $ssh_private_key_mode,
          owner       => 'pe-puppet',
          group       => 'pe-puppet',
          target_type => 'file',
          require     => Package['pe-puppetserver'],
        }
      }
    }

    class { 'pe_r10k':
      remote          => $r10k_remote,
      proxy           => $r10k_proxy,
      git_settings    => $r10k_git_config,
      forge_settings  => $r10k_forge_config,
      deploy_settings => $deploy_settings,
      r10k_basedir    => $environmentpath,
      postrun         => $r10k_postrun,
    }

    $_libssh_dir = '/opt/puppetlabs/server/data/puppetserver/.ssh'
    $_libssh_known_hosts = "${_libssh_dir}/known_hosts"

    file { $_libssh_dir:
      ensure  => 'directory',
      owner   => 'pe-puppet',
      group   => 'pe-puppet',
      mode    => '0700',
      require => Class[pe_r10k],
    }

    file { $_libssh_known_hosts:
      ensure  => 'file',
      owner   => 'pe-puppet',
      group   => 'pe-puppet',
      mode    => '0644',
      require => File[$_libssh_dir],
    }

    $r10k_known_hosts.each |$known_host| {
      $_rkh_ensure = $known_host['ensure'] ? {
        'absent' => 'absent',
        default  => 'present'
      }

      $_rkh_title = $known_host['title'] ? {
        undef   => "${known_host['name']}_${known_host['type']}",
        default => $known_host['title']
      }

      if $known_host['host_aliases'] {
        sshkey { "${_rkh_title}":
          ensure       => $_rkh_ensure,
          name         => "${known_host['name']}",
          type         => "${known_host['type']}",
          key          => "${known_host['key']}",
          host_aliases => $known_host['host_aliases'],
          target       => $_libssh_known_hosts,
          require      => File[$_libssh_known_hosts],
        }
      } else {
        sshkey { "${_rkh_title}":
          ensure  => $_rkh_ensure,
          name    => "${known_host['name']}",
          type    => "${known_host['type']}",
          key     => "${known_host['key']}",
          target  => $_libssh_known_hosts,
          require => File[$_libssh_known_hosts],
        }
      }
    }
  } else {
    # If the user did not specify a remote, then only include
    # the package and not the configuration. We don't want to
    # include the main class here as that prevents the user
    # from using a class resource to declare their configuration
    include pe_r10k::package
  }

  # File sync applies to all infra nodes
  if $file_sync_enabled or $code_manager_auto_configure {
    if $versioned_deploys_enabled == undef {
      warning("The `puppet_enterprise::profile::master::versioned_deploys` parameter is unset and so you are using the default value. The default will be changing in a future release, from `false` to `true`, and not using versioned_deploys is deprecated. Please see https://www.puppet.com/docs/pe/latest/lockless-code-deploys#lockless-code-deploys and migrate to versioned deploys at your earliest convenience.")
    }

    class {'puppet_enterprise::master::file_sync':
      certname                                  => $certname,
      puppet_master_host                        => $puppet_enterprise::puppet_master_host,
      master_of_masters_certname                => $primary_certname,
      puppetserver_jruby_puppet_master_code_dir => $puppetserver_codedir,
      puppet_default_codedir                    => $puppet_default_codedir,
      versioned_deploys_codedir                 => $versioned_deploys_codedir,
      puppetserver_webserver_ssl_port           => $ssl_listen_port,
      localcacert                               => $puppet_enterprise::params::localcacert,
      replication_mode                          => $replication_mode,
      provisioned_replicas                      => $provisioned_replicas,
      storage_service_disabled                  => !$is_primary,
      require                                   => [Package['pe-puppetserver'], File[$puppetserver_codedir]],
      versioned_deploys                         => $versioned_deploys_enabled,
      cert_data_discard_history                 => $cert_data_discard_history,
      versioned_sync_pool                       => $file_sync_versioned_sync_pool,
      copy_method                               => $file_sync_copy_method,
    }
  } else {
    class {'puppet_enterprise::master::file_sync_disabled':
      code_id_command      => $code_id_command,
      code_content_command => $code_content_command,
    }
  }
}
