# This profile sets up the PE ACE Server service.
# @param certname [String] The name of the service SSL certificate.
# @param run_service [Boolean] Should the service be running
# @param hostcrl [String] The host crl path
# @param ssl_listen_port [Boolean] The port used for SSL connections.
# @param master_host [String] The hostname of the primary.
# @param master_port [Integer] The port of puppet server on the primary.
# @param whitelist [Optional[Array[String]]] * Deprecated * Use allowlist instead.
# @param allowlist [Array[String]] Certificates that can be used to make requests to the service.
# @param ssl_cipher_suites [Array[String]] Cipher suites for TLS in preference order.
# @param service_loglevel [Enum[debug, info, error, warn]] Set service log level.
# @param settings [Hash] Any additional hocon settings to set in the service configuration
# @param license_check_interval_seconds [Integer] Number of seconds between checking the license for changes/updates
# @param license_check_retries [Integer] Number of times to retry fetching license data if the URL fails to return data.
# @param concurrency [Integer] Number of concurrent threads the ruby server uses to accept requests
class puppet_enterprise::profile::ace_server (
  Integer $ssl_listen_port                         = $puppet_enterprise::ace_server_port,
  String $certname                                 = $facts['clientcert'],
  String $master_host                              = lookup('pe_repo::compile_master_pool_address', { default_value => $puppet_enterprise::puppet_master_host }),
  Integer $master_port                             = $puppet_enterprise::puppet_master_port,
  Boolean $run_service                             = true,
  String  $hostcrl                                 = $puppet_enterprise::params::hostcrl,
  Optional[Array[String]] $whitelist               = undef,
  Array[String] $allowlist                         = pe_pick($whitelist, [$certname]),
  Array[String] $ssl_cipher_suites                 = $puppet_enterprise::ssl_cipher_suites_non_java,
  Enum[debug, info, error, warn] $service_loglevel = 'info',
  Integer $concurrency                             = $puppet_enterprise::ace_server_concurrency,
  Hash $settings                                   = {},
  Integer $license_check_interval_seconds          = $puppet_enterprise::license_check_interval_seconds,
  Integer $license_check_retries                   = $puppet_enterprise::license_check_retries
) {
  puppet_enterprise::deprecated_parameter { 'puppet_enterprise::profile::ace_server::whitelist': replacement => 'puppet_enterprise::profile::ace_server::allowlist' }
  $container = 'ace-server'

  puppet_enterprise::orchestrator::ruby_service { $container:
    container                      => $container,
    ssl_listen_port                => $ssl_listen_port,
    certname                       => $certname,
    run_service                    => $run_service,
    allowlist                      => $allowlist,
    ssl_cipher_suites              => $ssl_cipher_suites,
    service_loglevel               => $service_loglevel,
    license_api                    => "https://${puppet_enterprise::host_action_collector_host}:${puppet_enterprise::host_action_collector_port}/license/v1/current",
    license_check_interval_seconds => $license_check_interval_seconds,
    license_check_retries          => $license_check_retries,
    settings                       => {
      'concurrency'       => $concurrency,
      'cache-dir'         => "/opt/puppetlabs/server/data/${container}/cache",
      'puppet-server-uri' => "https://${master_host}:${master_port}",
      'ssl-ca-crls'       => $hostcrl,
    } + $settings,
  }
}
