# @param ssl_listen_port [Integer] The port that the proxy will be configured to listen on.
# @param enable_http_redirect [Boolean] If enabled, the proxy will redirect HTTP traffic on the specified port to HTTPS.
# @param ipv6_only [Boolean] If enabled, the proxy will only allow IPv6 connections.
# @param server_name [String] Which hostname to configure the proxy to listen for requests for
class puppet_enterprise::profile::console::proxy::http_redirect (
  Integer $ssl_listen_port,
  Boolean $enable_http_redirect = true,
  Boolean $ipv6_only = $puppet_enterprise::ipv6_only,
  String  $ssl_listen_address = $puppet_enterprise::params::ssl_address_url_safe,
  String  $server_name          = $puppet_enterprise::console_host,
  Puppet_enterprise::Replication_mode $replication_mode = 'none',
) {

  if $replication_mode != 'replica' {
    $http_redirect_file_ensure = $enable_http_redirect ? {
      true    => file,
      default => absent,
    }

    $redirect_return_base = '301 https://$server_name$request_uri'
    $redirect_return_port = $ssl_listen_port ? {
      443     => '',
      default => ":${ssl_listen_port}"
    }

    $_ipv6_only = $ipv6_only ? {
      true => 'on',
      false => 'off'
    }

    file { "${puppet_enterprise::nginx_conf_dir}/conf.d/http_redirect.conf":
      ensure  => $http_redirect_file_ensure,
      owner   => $puppet_enterprise::params::root_user,
      group   => $puppet_enterprise::params::root_group,
      mode    => '0644',
      content => epp('puppet_enterprise/console/http_redirect.conf.epp', {
        server_name  => $server_name,
        listen_address => $ssl_listen_address,
        return_value => "${redirect_return_base}${redirect_return_port}",
        ipv6_only    => $_ipv6_only,
      }),
      notify  => Service['pe-nginx'],
    }
  }

}
