# A define type to manage the creation of a read-only postgres users.
# In particular, it manages the necessary grants to enable such a user
# to have read-only access to any existing objects as well as changes
# the default access privileges so read-only access is maintained when
# new objects are created by the $db_owner
#
# @param user_name [String] The name of the postgres user
# @param database [String] The name of the database to grant access to.
# @param db_owner [String] The user which owns the database (i.e. the write user
#        for the database)
# @param tablespace_name [String] The name of the tablespace for this database.
# @param password_hash [String] The value of $_database_password in app_database.

define puppet_enterprise::puppetdb::read_only_user(
  String $user_name,
  String $database,
  String $db_owner,
  String $tablespace_name,
  Variant[String,Boolean] $password_hash = false,
  String $schema = 'public',
) {
  pe_postgresql::server::role { $user_name:
    password_hash => $password_hash,
    require       => Pe_postgresql::Server::Tablespace[$tablespace_name],
  } ->

  # Revoke a previous connect grant granted by the superuser
  # This is required until PE no longer supports upgrades from the 2021.7 LTS
  puppet_enterprise::psql { "REVOKE ${user_name} - CONNECT - ${database} granted by superuser":
    command => "REVOKE CONNECT ON DATABASE \"${database}\" FROM \"${user_name}\"",
    require => Pe_postgresql::Server::Role[$user_name],
    unless  => "SELECT * FROM pg_catalog.pg_database
                 WHERE datname = '${database}'
                 AND NOT datacl @> '\"${user_name}\"=c/\"pe-postgres\"'::aclitem"
  } ->

  # Grant the read role to the write role. This allows the write user to "bulldoze"
  # queries during PuppetDB's GC operations. It also is transitively granted to the
  # migrator user, which allows the migrator to kick out queries during an upgrade
  puppet_enterprise::psql { "GRANT ${user_name} TO ${db_owner}":
    command => "GRANT \"${user_name}\" TO \"${db_owner}\"",
    require => Pe_postgresql::Server::Db[$database],
    unless => "SELECT * FROM pg_has_role(
                '${db_owner}',
                '${user_name}',
                'member')
              WHERE pg_has_role = true"
  } ->

  puppet_enterprise::pg::default_read_grant {"${database} grant read perms on new objects from ${db_owner} to ${user_name}":
    table_creator => $db_owner,
    table_reader  => $user_name,
    database      => $database,
    schema        => 'public',
    require       => Pe_postgresql::Server::Role[$user_name],
  } ->

  puppet_enterprise::psql {"${title}/default_functions_grant/sql":
    db      => $database,
    command => "ALTER DEFAULT PRIVILEGES
                  FOR USER \"${db_owner}\"
                  IN SCHEMA \"${schema}\"
                GRANT EXECUTE ON FUNCTIONS
                  TO \"${user_name}\"",
    unless  => "SELECT
                  ns.nspname,
                  acl.defaclobjtype,
                  acl.defaclacl
                FROM pg_default_acl acl
                JOIN pg_namespace ns ON acl.defaclnamespace=ns.oid
                WHERE acl.defaclacl::text ~ '.*\\\\\"${user_name}\\\\\"=X/\\\\\"${db_owner}\\\\\".*'
                AND nspname = '${schema}'",
  } ->

  puppet_enterprise::psql {"${title}/default_sequences_grant/sql":
    db      => $database,
    command => "ALTER DEFAULT PRIVILEGES
                  FOR USER \"${db_owner}\"
                  IN SCHEMA \"${schema}\"
                GRANT USAGE ON SEQUENCES
                  TO \"${user_name}\"",
    unless  => "SELECT
                  ns.nspname,
                  acl.defaclobjtype,
                  acl.defaclacl
                FROM pg_default_acl acl
                JOIN pg_namespace ns ON acl.defaclnamespace=ns.oid
                WHERE acl.defaclacl::text ~ '.*\\\\\"${user_name}\\\\\"=U/\\\\\"${db_owner}\\\\\".*'
                AND nspname = '${schema}'",
  } ->

  puppet_enterprise::pg::read_grant {"${database} grant read-only perms on existing objects to ${user_name}":
    table_reader => $user_name,
    database     => $database,
    schema       => 'public',
    require      => Pe_postgresql::Server::Role[$user_name],
  }

}
