# puppetlabs-edgeops

[![PR Checks](https://github.com/puppetlabs/puppetlabs-edgeops/actions/workflows/pr-checks.yml/badge.svg)](https://github.com/puppetlabs/puppetlabs-edgeops/actions/workflows/pr-checks.yml)

The official Puppet module for managing network devices via NETCONF protocol.

## Table of Contents

1. [Description](#description)
2. [Requirements](#requirements)
3. [Installation](#installation)
4. [Quick Start](#quick-start)
5. [Available Tasks](#available-tasks)
6. [Usage Examples](#usage-examples)
7. [Writing Custom Tasks](#writing-custom-tasks)
8. [Configuration](#configuration)
9. [Security & Sensitive Data](#security--sensitive-data)
10. [API Reference](#api-reference)
11. [Troubleshooting](#troubleshooting)
12. [Support](#support)

## Description

The puppetlabs-edgeops module provides enterprise-grade tools for managing edge network devices with Puppet Bolt. Built on a robust Ruby implementation of the NETCONF protocol, this module enables:

- **Configuration Management** - Retrieve, modify, and validate device configurations
- **Operational Tasks** - Execute NETCONF RPC operations and custom commands
- **Lock Management** - Intelligent handling of configuration locks and recovery
- **Schema Discovery** - Explore device capabilities and YANG models
- **Enterprise Scale** - Manage thousands of devices using Bolt orchestration

## Requirements

### License Requirements

⚠️ **This module requires a valid Puppet Enterprise license with Edge entitlements.**

Available through:
- **Puppet Bolt** - Standalone automation (requires Edge entitlement)
- **Puppet Enterprise** - Full orchestration (requires PE Advanced with Edge)

### System Requirements

- **Puppet Bolt** 5.0 or higher
- **Ruby** 3.1 or higher (included with Bolt)
- **Target Devices** must support NETCONF protocol (RFC 6241)

### Supported Devices

Tested and verified on:
- Arista EOS

Expected to work with any RFC 6241 compliant NETCONF implementation (not yet tested):
- Cisco IOS-XR / IOS-XE  
- Juniper Junos
- Other NETCONF-enabled devices

## Installation

### 1. Create Bolt Project

```bash
mkdir my-network-automation
cd my-network-automation
bolt project init
```

### 2. Configure Module

Add to `bolt-project.yaml`:

```yaml
name: my_network_project
modules:
  - name: puppetlabs/edgeops
    version_requirement: '>=1.0.0 <2.0.0'
```

### 3. Configure Authentication

Add to `bolt-project.yaml`:

```yaml
module-install:
  forge:
    authorization_token: Bearer <your-api-key>
    baseurl: https://forge.puppet.com
```

> **Note**: Contact Puppet Sales for API key and entitlement verification.

### 4. Install Module

```bash
bolt module install
```

## Quick Start

### 1. Create Inventory

Create `inventory.yaml`:

```yaml
targets:
  - name: switch1
    uri: 172.16.0.1
    config:
      transport: remote
      remote:
        port: 830
        user: admin
        password: admin123
        host-key-check: false
```

### 2. Test Connection

```bash
# Discover device capabilities
bolt task run edgeops::netconf_discover --targets switch1

# Get current configuration
bolt task run edgeops::netconf_get_config --targets switch1
```

## Available Tasks

### Discovery & Information

| Task | Description | Key Parameters |
|------|-------------|----------------|
| `netconf_discover` | Comprehensive device discovery | `action`: auto, capabilities, schemas |
| `netconf_list_capabilities` | List NETCONF capabilities | - |
| `netconf_list_schemas` | List available YANG schemas | - |
| `netconf_get_schema` | Download specific YANG schema | `identifier`: schema name |
| `netconf_get_interfaces` | Get interface information | `interface`: specific interface |

### Configuration Management

| Task | Description | Key Parameters |
|------|-------------|----------------|
| `netconf_get_config` | Retrieve configuration | `source`: running, candidate, startup<br>`filter`: XML filter |
| `netconf_edit_config` | Modify configuration | `target`: running, candidate<br>`config`: XML config<br>`operation`: merge, replace |
| `netconf_discard_changes` | Discard uncommitted changes | - |

### Lock Management

| Task | Description | Key Parameters |
|------|-------------|----------------|
| `netconf_lock` | Lock a datastore | `datastore`: running, candidate, startup |
| `netconf_unlock` | Unlock a datastore | `datastore`: running, candidate, startup |
| `netconf_recover_lock` | Smart lock recovery | `datastore`: target datastore<br>`force`: kill blocking sessions |
| `netconf_kill_session` | Terminate NETCONF session | `session_id`: session to kill |

## Usage Examples

### Basic Configuration Retrieval

```bash
# Get running configuration
bolt task run edgeops::netconf_get_config --targets switch1

# Get specific interface configuration
bolt task run edgeops::netconf_get_config --targets switch1 \
  filter='<interfaces><interface><name>eth0</name></interface></interfaces>'

# Get candidate configuration
bolt task run edgeops::netconf_get_config --targets switch1 \
  source=candidate
```

### Configuration Changes

```bash
# Edit interface description
bolt task run edgeops::netconf_edit_config --targets switch1 \
  config='<config>
    <interfaces xmlns="http://openconfig.net/yang/interfaces">
      <interface>
        <name>Ethernet1</name>
        <config>
          <description>Uplink to Core</description>
        </config>
      </interface>
    </interfaces>
  </config>'

# Replace entire configuration section
bolt task run edgeops::netconf_edit_config --targets switch1 \
  operation=replace \
  config='<config>...</config>'
```

### Lock Recovery Workflow

When encountering lock issues, use this escalation path:

```bash
# 1. Smart recovery (recommended first step)
bolt task run edgeops::netconf_recover_lock --targets switch1

# 2. If uncommitted changes are blocking (session ID 0)
bolt task run edgeops::netconf_discard_changes --targets switch1

# 3. Force recovery by killing blocking sessions
bolt task run edgeops::netconf_recover_lock --targets switch1 force=true

# 4. Manual session termination (last resort)
bolt task run edgeops::netconf_kill_session --targets switch1 session_id=12345
```

The `netconf_recover_lock` task automatically:
- Identifies lock type (session ID 0 vs active session)
- Takes appropriate recovery action
- Provides clear feedback and recommendations

### Sensitive Data Protection

```bash
# Redact passwords and secrets from output
bolt task run edgeops::netconf_get_config --targets switch1 \
  redact_patterns='["password","secret","key","community"]'

# Use default redaction patterns
bolt task run edgeops::netconf_get_config --targets switch1 \
  include_default_patterns=true
```

## Writing Custom Tasks

### Task Structure

Create task metadata (`tasks/my_task.json`):

```json
{
  "description": "My custom NETCONF task",
  "input_method": "stdin",
  "files": [
    "edgeops/files/task_helper.rb",
    "edgeops/lib/puppet_x/puppetlabs/netconf/client.rb",
    "edgeops/lib/puppet_x/puppetlabs/netconf/session.rb",
    "edgeops/lib/puppet_x/puppetlabs/netconf/redactor.rb"
  ],
  "remote": false,
  "parameters": {
    "interface": {
      "description": "Interface to configure",
      "type": "String"
    },
    "source_datastore": {
      "description": "Source datastore override",
      "type": "Optional[Enum[running, candidate, startup]]"
    }
  }
}
```

Create task implementation (`tasks/my_task.rb`):

```ruby
#!/usr/bin/env ruby
require_relative '../../edgeops/files/task_helper.rb'
require 'puppet_x/puppetlabs/netconf/session'
require 'json'

result = PuppetX::Puppetlabs::Netconf::Session.with_session do |session|
  # Access parameters
  interface = session.task_params['interface']
  
  # Get configuration
  config = session.get_config
  
  # Make changes
  new_config = "<config>...</config>"
  session.edit_config(new_config)
  
  # Commit if using candidate
  session.commit if session.supports_candidate?
  
  # Report results
  session.report_result({
    'status' => 'success',
    'interface' => interface,
    'message' => 'Configuration updated'
  })
end

# Output result as JSON (required!)
puts result.to_json
```

### Key Points

- **Always declare required files** in task metadata
- **Capture the session result**: `result = Session.with_session`
- **Output JSON**: `puts result.to_json`
- **Use session helpers**: `session.supports_candidate?`, `session.task_params`

## Configuration

### Inventory Configuration

```yaml
targets:
  - name: router1
    uri: 192.168.1.1
    config:
      transport: remote
      remote:
        port: 830
        user: admin
        password: secret
        host-key-check: false
        # Timeout configuration
        timeout: 30              # Global default
        connect-timeout: 10      # Connection timeout
        command-timeout: 60      # Operation timeout
        hello-timeout: 5         # NETCONF hello timeout
        # Protocol settings
        keepalive-interval: 30
        protocol-version: '1.1'  # Force specific version
```

### Session Parameters

Control session behavior through task parameters:

- `source_datastore` - Default source for get operations (default: 'running')
- `target_datastore` - Default target for edit operations (default: 'running')
- `redact_patterns` - Patterns to redact from output
- `include_default_patterns` - Include built-in redaction patterns

### Parameter Precedence

Session options in `with_session()` always override task parameters, ensuring reliable behavior:

```ruby
PuppetX::Puppetlabs::Netconf::Session.with_session(
  source_datastore: 'candidate',   # Enforced by task
  target_datastore: 'running'      # Enforced by task
) do |session|
  # Uses candidate->running regardless of user input
  config = session.get_config
  session.edit_config(config)
end
```

**Precedence:** Session options > Task parameters

This ensures custom tasks can enforce specific behavior without user override.

## Security & Sensitive Data

### Redaction Control

Redaction is **enabled by default** to protect sensitive data in logs. No configuration required for basic protection.

**Control Options:**

```bash
# Use defaults only (recommended)
bolt task run edgeops::netconf_get_config --targets devices

# Add custom patterns while keeping defaults
bolt task run edgeops::netconf_get_config --targets devices \
  redact_patterns='["custom_secret", "api_key"]'

# Use only custom patterns
bolt task run edgeops::netconf_get_config --targets devices \
  redact_patterns='["only_this"]' include_default_patterns=false

# Disable redaction (NOT recommended)
bolt task run edgeops::netconf_get_config --targets devices \
  include_default_patterns=false
```

**Default Patterns:** `password`, `secret`, `key`, `token`, `credential`, `auth`, `community`, `private`, `encrypted`

⚠️ **Warning:** Disabling redaction exposes sensitive data in logs.

### Best Practices

1. **Always use redaction** for production devices
2. **Don't log sensitive values** in custom results
3. **Test redaction** in development first
4. **Use vendor-specific patterns** as needed

## API Reference

### Session Methods

| Method | Description | Parameters |
|--------|-------------|------------|
| `get_config` | Get configuration | `filter`, `source` |
| `edit_config` | Edit configuration | `config`, `target`, `operation` |
| `get` | Get operational data | `filter` |
| `lock` | Lock datastore | `datastore` |
| `unlock` | Unlock datastore | `datastore` |
| `commit` | Commit changes | `confirmed`, `timeout` |
| `discard_changes` | Discard changes | - |
| `validate` | Validate config | `source` |
| `copy_config` | Copy configuration | `source`, `target` |
| `delete_config` | Delete configuration | `target` |
| `kill_session` | Kill session | `session_id` |

### Capability Helpers

| Method | Description | Returns |
|--------|-------------|---------|
| `supports_candidate?` | Candidate datastore support | Boolean |
| `supports_startup?` | Startup datastore support | Boolean |
| `supports_writable_running?` | Direct running edits | Boolean |
| `supports_validate?` | Validation support | Boolean |
| `supports_rollback_on_error?` | Rollback capability | Boolean |
| `supports_confirmed_commit?` | Confirmed commits | Boolean |

### Error Classes

All errors inherit from `NetconfError` base class:

- **Connection Errors**
  - `OpenError` - Connection failed
  - `AuthenticationError` - Authentication failed
  - `TimeoutError` - Operation timeout

- **Protocol Errors**
  - `ProtocolError` - Protocol violation
  - `MessageIdMismatch` - Response mismatch
  - `MalformedMessage` - Invalid XML

- **Operation Errors** (RFC 6241)
  - `LockDenied` - Lock unavailable
  - `InvalidValue` - Invalid parameter
  - `AccessDenied` - Permission denied
  - `OperationNotSupported` - Unsupported operation
  - `DataExists` / `DataMissing` - Data conflicts

## Troubleshooting

### Common Issues

#### Module Installation

**"Module not found on Forge"**
- Verify entitlement with Puppet Support
- Check API key configuration
- Ensure correct Forge URL

**"Authentication failed"**
- Verify API key format: `Bearer <key>`
- Check key hasn't expired
- Confirm entitlement is active

#### Connection Issues

**"Connection refused"**
- Verify NETCONF is enabled (usually port 830)
- Check firewall rules
- Confirm device IP and port

**"Authentication failed"**
- Verify username/password
- Check SSH key permissions (600)
- Confirm user has NETCONF access

**"Protocol version mismatch"**
- Force version: `protocol-version: '1.0'`
- Check device capabilities
- Verify NETCONF implementation

#### Lock Issues

**"Lock denied by session ID 0"**
- Uncommitted changes present
- Run: `bolt task run edgeops::netconf_discard_changes`

**"Lock denied by session X"**
- Another session holds lock
- Run: `bolt task run edgeops::netconf_recover_lock`

### Debug Logging

Enable debug output:

```bash
bolt task run edgeops::netconf_get_config --targets device --log-level debug
```

Check Bolt logs:
- `~/.puppetlabs/bolt/logs/`

## Support

### Getting Help

- **Sales & Licensing**: [Puppet Sales](https://puppet.com/contact-sales/)
- **Technical Support**: [Customer Support Portal](https://support.puppet.com)
- **Documentation**: [Puppet Docs](https://puppet.com/docs)
- **Community**: [Puppet Community Slack](https://slack.puppet.com)

### Reporting Issues

1. Check [Troubleshooting](#troubleshooting) section
2. Search existing issues on GitHub
3. Contact Puppet Support (Enterprise customers)

### Contributing

This module is maintained by Puppet, Inc. For contribution guidelines and development setup, see CONTRIBUTING.md.

## License

Copyright 2024 Puppet, Inc.

Licensed under the Apache License, Version 2.0.