# frozen_string_literal: true

require 'net/ssh'
require 'rexml/document'
require 'rexml/xpath'
require 'logger'

# The main namespace for Puppet extensions.
#
# This module contains all Puppet extensions that extend the core
# functionality of Puppet. These extensions are typically specific
# to particular vendors or technologies.
# Puppetlabs-specific Puppet extensions.
#
# This module contains extensions developed and maintained by Puppet Inc.
# (formerly Puppet Labs) for extending Puppet's capabilities with
# vendor-specific or technology-specific functionality.
# NETCONF protocol implementation for network device management.
#
# This module provides a complete implementation of the NETCONF protocol
# as defined in RFC 6241, enabling programmatic configuration and
# management of network devices.
#
# @see https://tools.ietf.org/html/rfc6241 NETCONF Protocol RFC 6241
# @since 1.0.0
module PuppetX
  module Puppetlabs
    module Netconf
      # Namespace for NETCONF-specific exceptions.
      #
      # All NETCONF operations may raise these exceptions to indicate
      # various error conditions. Applications should handle these
      # exceptions appropriately.
      #
      # @example Handling NETCONF errors
      #   begin
      #     client.get_config('running')
      #   rescue NetconfError::RpcError => e
      #     logger.error "RPC failed: #{e.message}"
      #     logger.error "Error type: #{e.error_type}"
      #     logger.error "Error tag: #{e.error_tag}"
      #   rescue NetconfError::TimeoutError => e
      #     logger.error "Operation timed out after #{e.timeout_value}s"
      #   end
      module NetconfError
        # Base error class for all NETCONF errors.
        #
        # This class provides the foundation for all NETCONF-specific
        # exceptions, allowing applications to catch all NETCONF errors
        # with a single rescue clause.
        #
        # @attr_reader [Hash] error_info Additional error context
        class NetconfError < StandardError
          attr_reader :error_info

          # Creates a new NetconfError.
          #
          # @param message [String] Human-readable error message
          # @param error_info [Hash] Additional error context
          # @option error_info [String] :session_id NETCONF session ID
          # @option error_info [String] :message_id RPC message ID
          # @option error_info [String] :operation The operation that failed
          def initialize(message, error_info = {})
            super(message)
            @error_info = error_info
          end
        end

        # Raised when an operation is attempted in an invalid state.
        # @example Attempting to send RPC on closed connection
        #   client.close
        #   client.get_config # raises StateError
        class StateError < NetconfError; end

        # Raised when a connection cannot be established.
        class OpenError < NetconfError; end

        # Raised when the connection is unexpectedly closed.
        class ConnectionClosed < NetconfError; end

        # Raised when authentication fails.
        # @note Check credentials and SSH key configuration
        class AuthenticationError < OpenError; end

        # Raised when host key verification fails.
        # @note Check known_hosts or disable host-key-check if appropriate
        class HostKeyError < OpenError; end

        # Raised when an operation times out.
        #
        # Different operations have different timeout values that can be
        # configured when creating the client.
        #
        # @attr_reader [Symbol] timeout_type The type of timeout (:connect, :command, :hello, :idle)
        # @attr_reader [Integer] timeout_value The timeout value in seconds
        class TimeoutError < NetconfError
          attr_reader :timeout_type, :timeout_value

          # Creates a new TimeoutError.
          #
          # @param message [String] Error message
          # @param timeout_type [Symbol] Type of timeout that occurred
          # @param timeout_value [Integer] The timeout value that was exceeded
          def initialize(message, timeout_type, timeout_value)
            super(message)
            @timeout_type = timeout_type
            @timeout_value = timeout_value
          end
        end

        # Protocol errors
        class ProtocolError < NetconfError; end
        class MessageIdMismatch < ProtocolError; end
        class InvalidResponse < ProtocolError; end
        class MissingCapability < ProtocolError; end

        # RPC errors - base class for all NETCONF RPC errors
        # Preserves complete error context from the device
        class RpcError < NetconfError
          attr_reader :error_type, :error_tag, :error_severity, :error_info, :error_path, :error_app_tag, :error_details

          def initialize(message, error_details = {})
            super(message)
            if error_details.is_a?(Hash)
              @error_type = error_details[:type]
              @error_tag = error_details[:tag]
              @error_severity = error_details[:severity]
              @error_info = error_details[:info]
              @error_path = error_details[:path]
              @error_app_tag = error_details[:app_tag]
              @error_details = error_details # Keep ALL details including vendor-specific fields
            else
              # Minimal details provided (backwards compatibility)
              @error_type = error_details
              @error_details = { type: error_details }
            end
          end

          # Access any error detail field
          def [](key)
            @error_details ? @error_details[key] : nil
          end

          # Check if a specific detail field exists
          def detail?(key)
            @error_details&.key?(key) || false
          end
        end

        # Specific RPC operation errors based on RFC 6241 error-tags

        # Resource/Lock related errors
        class ResourceInUse < RpcError; end           # error-tag: in-use
        class LockDenied < RpcError; end              # error-tag: lock-denied
        class ResourceDenied < RpcError; end          # error-tag: resource-denied

        # Value/Attribute errors
        class InvalidValue < RpcError; end            # error-tag: invalid-value
        class MissingAttribute < RpcError; end        # error-tag: missing-attribute
        class BadAttribute < RpcError; end            # error-tag: bad-attribute
        class UnknownAttribute < RpcError; end        # error-tag: unknown-attribute

        # Element errors
        class MissingElement < RpcError; end          # error-tag: missing-element
        class BadElement < RpcError; end              # error-tag: bad-element
        class UnknownElement < RpcError; end          # error-tag: unknown-element
        class UnknownNamespace < RpcError; end        # error-tag: unknown-namespace

        # Data errors
        class DataExists < RpcError; end              # error-tag: data-exists
        class DataMissing < RpcError; end             # error-tag: data-missing

        # Operation errors
        class OperationNotSupported < RpcError; end   # error-tag: operation-not-supported
        class OperationFailed < RpcError; end         # error-tag: operation-failed
        class PartialOperation < RpcError; end        # error-tag: partial-operation
        class MalformedMessage < RpcError; end        # error-tag: malformed-message

        # Other errors
        class AccessDenied < RpcError; end            # error-tag: access-denied
        class RollbackFailed < RpcError; end          # error-tag: rollback-failed
        class TooBig < RpcError; end                  # error-tag: too-big

        # Configuration errors
        class ConfigurationError < NetconfError; end
      end

      # NETCONF Client implementation for managing network devices.
      #
      # This class provides a complete NETCONF client implementation supporting
      # both NETCONF 1.0 (EOM framing) and 1.1 (chunked framing) protocols.
      # It handles SSH transport, authentication, RPC operations, and automatic
      # reconnection with comprehensive timeout management.
      #
      # @example Basic usage
      #   client = PuppetX::Puppetlabs::Netconf::Client.new('router.example.com', 'admin',
      #     password: 'secret',
      #     port: 830
      #   )
      #   client.connect
      #   config = client.rpc.get_config('running')
      #   client.disconnect
      #
      # @example With timeout configuration
      #   client = PuppetX::Puppetlabs::Netconf::Client.new('router.example.com', 'admin',
      #     password: 'secret',
      #     connect_timeout: 30,      # 30s to establish connection
      #     command_timeout: 60,      # 60s for each command
      #     idle_timeout: 300,        # Close connection after 5 minutes idle
      #     keepalive_interval: 30    # Send keepalive every 30s
      #   )
      #
      # @example With automatic reconnection
      #   client = PuppetX::Puppetlabs::Netconf::Client.new('router.example.com', 'admin',
      #     password: 'secret',
      #     auto_reconnect: true,     # Enable automatic reconnection
      #     reconnect_attempts: 3,    # Try 3 times
      #     reconnect_delay: 5        # Wait 5s between attempts
      #   )
      #
      # == Timeout Parameters
      #
      # The client supports multiple timeout parameters for fine-grained control:
      #
      # * +:timeout+ - General timeout (default: 10s). Deprecated - use specific timeouts instead.
      # * +:connect_timeout+ - Time to establish SSH connection (default: same as timeout).
      # * +:disconnect_timeout+ - Time to gracefully close connection (default: 5s).
      # * +:command_timeout+ - Time for individual NETCONF commands (default: timeout * 2).
      # * +:hello_timeout+ - Time for initial NETCONF hello exchange (default: same as timeout).
      # * +:idle_timeout+ - Close connection after this many seconds of inactivity (default: timeout * 6).
      #
      # == Keepalive and Monitoring
      #
      # * +:keepalive_interval+ - Send keepalive messages at this interval (default: 30s).
      #   Set to 0 to disable keepalives. Keepalives help detect broken connections
      #   and prevent firewalls from closing idle connections.
      #
      # The monitoring thread runs when either keepalive or idle_timeout is enabled.
      # It checks for idle timeout and sends keepalive messages as needed.
      #
      # == Reconnection Parameters
      #
      # * +:auto_reconnect+ - Automatically reconnect on connection errors (default: false).
      # * +:reconnect_attempts+ - Number of reconnection attempts (default: 3).
      # * +:reconnect_delay+ - Seconds to wait between attempts (default: 5).
      #
      # @since 1.0.0
      class Client
        NETCONF_PORT = 830
        NETCONF_SUBSYSTEM = 'netconf'
        NAMESPACE = 'urn:ietf:params:xml:ns:netconf:base:1.0'
        MSG_END = ']]>]]>'
        MSG_END_RE = %r{\]\]>\]\]>[\r\n]*$}.freeze
        MSG_HELLO = <<-EOM.gsub(%r{\s+\|}, '')
          |<?xml version="1.0" encoding="UTF-8"?>
          |<hello xmlns="urn:ietf:params:xml:ns:netconf:base:1.0">
          |  <capabilities>
          |    <capability>urn:ietf:params:netconf:base:1.1</capability>
          |    <capability>urn:ietf:params:netconf:base:1.0</capability>
          |  </capabilities>
          |</hello>
        EOM

        # Chunked framing constants for NETCONF 1.1
        CHUNK_SIZE = 16_384 # 16KB chunks
        CHUNK_HEADER_RE = %r{\n#(\d+)\n}.freeze
        CHUNK_END = "\n##\n"

        attr_reader :capabilities, :session_id, :server_capabilities, :state, :protocol_version

        # Initialize a new NETCONF client.
        #
        # @param host [String] The hostname or IP address of the NETCONF device
        # @param username [String] The username for authentication
        # @param options [Hash] Connection and behavior options
        # @option options [Integer] :port (830) The NETCONF port
        # @option options [String] :password The password for authentication
        # @option options [String] :private_key Path to private key file for key-based auth
        # @option options [Integer] :timeout (10) General timeout in seconds (deprecated)
        # @option options [Integer] :connect_timeout SSH connection timeout
        # @option options [Integer] :disconnect_timeout (5) Graceful disconnect timeout
        # @option options [Integer] :command_timeout Individual command timeout
        # @option options [Integer] :hello_timeout NETCONF hello exchange timeout
        # @option options [Integer] :idle_timeout Connection idle timeout
        # @option options [Integer] :keepalive_interval (30) Keepalive interval in seconds
        # @option options [Boolean] :auto_reconnect (false) Enable automatic reconnection
        # @option options [Integer] :reconnect_attempts (3) Number of reconnection attempts
        # @option options [Integer] :reconnect_delay (5) Delay between reconnection attempts
        # @option options [Symbol] :verify_host_key (:always) SSH host key verification
        # @option options [String] :force_version Force specific NETCONF version ('1.0' or '1.1')
        #
        # @raise [ArgumentError] If required parameters are missing
        def initialize(host, username, options = {})
          @host = host
          @username = username
          @options = {
            port: NETCONF_PORT,
            timeout: 10,               # General timeout (deprecated, use specific timeouts)
            connect_timeout: nil,      # SSH connection timeout
            disconnect_timeout: nil,   # Graceful disconnect timeout
            command_timeout: nil,      # Individual command timeout
            hello_timeout: nil,        # Hello exchange timeout
            idle_timeout: nil          # Idle connection timeout
          }.merge(options)

          # Apply general timeout as default for specific timeouts if not set
          @options[:connect_timeout] ||= @options[:timeout]
          @options[:disconnect_timeout] ||= 5 # 5s default for disconnect
          @options[:command_timeout] ||= @options[:timeout] * 2 # Commands may take longer
          @options[:hello_timeout] ||= @options[:timeout]
          @options[:idle_timeout] = @options[:timeout] * 6 if @options[:idle_timeout].nil? # 60s default for idle

          @state = :NETCONF_CLOSED
          @rpc_message_id = 1
          @capabilities = []
          @server_capabilities = []
          @session_id = nil
          @transaction = {}
          @protocol_version = '1.0' # Default to 1.0, will be negotiated during connect
          @force_version = @options[:force_version] # Allow forcing a specific version
          @keepalive_thread = nil
          @keepalive_interval = @options[:keepalive_interval] || 30 # Default 30 seconds
          @last_activity = Time.now
          @auto_reconnect = @options[:auto_reconnect] || false
          @reconnect_attempts = @options[:reconnect_attempts] || 3
          @reconnect_delay = @options[:reconnect_delay] || 5 # seconds
          @redactor = @options[:redactor] # Optional redactor for sanitizing logs
        end

        # Public accessor for the redactor instance
        attr_reader :redactor

        def connect
          logger = get_logger
          logger&.debug("NETCONF connect starting for #{@host}")

          raise NetconfError::StateError, 'Already connected' if @state == :NETCONF_OPEN

          transaction_open # This now raises exceptions on failure

          logger.trace('NETCONF waiting for server hello')
          hello_xml = transaction_receive_hello
          safe_log_xml(logger, :trace, 'NETCONF server hello received:', hello_xml)

          hello_rsp = REXML::Document.new(hello_xml)

          # Use namespace-aware XPath for parsing capabilities
          @capabilities = REXML::XPath.match(hello_rsp, '//nc:capability', 'nc' => NAMESPACE).map(&:text)
          @server_capabilities = @capabilities
          @session_id = REXML::XPath.first(hello_rsp, '//nc:session-id', 'nc' => NAMESPACE).text

          # Validate that server supports base NETCONF capability
          unless @server_capabilities.any? { |cap| cap.include?('urn:ietf:params:netconf:base:1.0') }
            raise NetconfError::MissingCapability, 'Server does not support required NETCONF base:1.0 capability'
          end

          # Negotiate protocol version
          @protocol_version = negotiate_protocol_version(@server_capabilities)

          logger.debug("NETCONF session-id: #{@session_id}")
          logger.debug("NETCONF protocol version: #{@protocol_version}")
          logger.debug("NETCONF server capabilities count: #{@capabilities.count}")
          logger.trace("NETCONF server capabilities: #{@capabilities.join(', ')}")

          logger.trace('NETCONF sending client hello')
          transaction_send_hello

          @state = :NETCONF_OPEN
          logger.debug('NETCONF connection established')

          # Start monitoring thread if keepalive or idle timeout is enabled
          start_monitoring if @keepalive_interval > 0 || @options[:idle_timeout]

          true
        end

        def disconnect
          return unless @state == :NETCONF_OPEN

          # Stop monitoring thread
          stop_monitoring

          # Send close-session RPC for graceful termination with timeout
          begin
            logger = get_logger
            logger&.debug('NETCONF sending close-session')

            # Use disconnect timeout for the close-session operation
            Timeout.timeout(@options[:disconnect_timeout]) do
              # Build and send close-session RPC
              close_doc = REXML::Document.new('<rpc><close-session/></rpc>')
              close_rpc = close_doc.root
              close_rpc.add_namespace(NAMESPACE)
              close_rpc.add_attribute('message-id', @rpc_message_id.to_s)

              # Send the close-session request
              send_and_receive(close_doc.to_s)

              logger&.debug('NETCONF close-session sent successfully')
            end
          rescue Timeout::Error
            # Disconnect timeout exceeded
            logger&.warn("NETCONF close-session timed out after #{@options[:disconnect_timeout]}s")
          rescue => e
            # Log but don't fail - we still want to close the transport
            logger&.warn("NETCONF close-session failed: #{e.message}")
            logger&.debug("NETCONF close-session error class: #{e.class.name}")
            logger&.debug("NETCONF close-session backtrace: #{e.backtrace[0..3].join("\n")}")
          end

          transaction_close
          @state = :NETCONF_CLOSED
          true
        end

        def open?
          @state == :NETCONF_OPEN
        end

        def closed?
          @state == :NETCONF_CLOSED
        end

        def rpc
          @rpc_executor ||= RpcExecutor.new(self)
        end

        def send_and_receive(cmd_str)
          logger = get_logger
          safe_log_xml(logger, :trace, 'NETCONF sending XML:', cmd_str)

          # Update last activity time for keepalive
          @last_activity = Time.now

          attempts = 0
          begin
            if @protocol_version == '1.1'
              # Use chunked framing for NETCONF 1.1
              encoded_msg = encode_chunked(cmd_str)
              transaction_send(encoded_msg)
            else
              # Use EOM delimiter for NETCONF 1.0
              transaction_send(cmd_str)
              transaction_send(MSG_END)
            end

            response = transaction_receive
            safe_log_xml(logger, :trace, 'NETCONF received XML:', response)

            response
          rescue => e
            raise e unless @auto_reconnect && attempts < @reconnect_attempts && reconnectable_error?(e)
            attempts += 1
            logger.warn("NETCONF connection error: #{e.message}, attempting reconnect (#{attempts}/#{@reconnect_attempts})")

            # Try to reconnect
            if attempt_reconnect
              logger.info('NETCONF reconnection successful, retrying operation')
              retry
            else
              logger.error('NETCONF reconnection failed')
              raise e
            end
          end
        end

        def rpc_exec(cmd_nx)
          logger = get_logger
          raise NetconfError::StateError, 'Not connected' unless @state == :NETCONF_OPEN

          # cmd_nx should be either the rpc element or an operation element
          # If it's already an rpc element, use it directly
          # Otherwise, it's an operation element and we need to get the parent rpc
          logger.debug("rpc_exec called with element: #{cmd_nx.name}")
          rpc_nx = if cmd_nx.name == 'rpc'
                     cmd_nx
                   else
                     cmd_nx.parent
                   end

          # Add namespace and message-id to the RPC element
          rpc_nx.add_namespace(NAMESPACE)
          message_id = @rpc_message_id.to_s
          rpc_nx.add_attribute('message-id', message_id)

          logger.debug("NETCONF RPC message-id: #{message_id}")
          @rpc_message_id += 1

          # Log the full RPC being sent (with redaction)
          rpc_xml = rpc_nx.to_s
          safe_log_xml(logger, :info, 'NETCONF RPC request XML:', rpc_xml)

          response_xml = send_and_receive(rpc_xml)

          rsp_doc = REXML::Document.new(response_xml)
          rsp_nx = rsp_doc.root

          # Validate message-id matches
          response_message_id = rsp_nx.attributes['message-id']
          if response_message_id != message_id
            logger.error("NETCONF message-id mismatch: sent '#{message_id}', received '#{response_message_id}'")
            raise NetconfError::MessageIdMismatch, "Message-id mismatch: expected '#{message_id}', got '#{response_message_id}'"
          end

          # Check for errors
          rpc_errs = REXML::XPath.match(rsp_nx, '//nc:rpc-error', 'nc' => NAMESPACE)
          if rpc_errs.count.positive?
            logger.error("NETCONF RPC response contains #{rpc_errs.count} error(s), handling first error") if rpc_errs.count > 1
            logger.error('NETCONF RPC response contains 1 error') if rpc_errs.count == 1

            # Log raw error XML for debugging
            rpc_errs.each do |err|
              safe_log_xml(logger, :debug, 'Raw error XML:', err)
            end

            # Process each error and collect details
            error_details = rpc_errs.map do |err|
              details = {}

              # Extract all standard NETCONF error fields
              details[:severity] = REXML::XPath.first(err, 'nc:error-severity', 'nc' => NAMESPACE)&.text
              details[:tag] = REXML::XPath.first(err, 'nc:error-tag', 'nc' => NAMESPACE)&.text
              details[:message] = REXML::XPath.first(err, 'nc:error-message', 'nc' => NAMESPACE)&.text
              details[:type] = REXML::XPath.first(err, 'nc:error-type', 'nc' => NAMESPACE)&.text
              details[:path] = REXML::XPath.first(err, 'nc:error-path', 'nc' => NAMESPACE)&.text
              details[:app_tag] = REXML::XPath.first(err, 'nc:error-app-tag', 'nc' => NAMESPACE)&.text

              # Extract error-info which can contain vendor-specific details
              error_info_elem = REXML::XPath.first(err, 'nc:error-info', 'nc' => NAMESPACE)
              if error_info_elem
                # Preserve structured error-info content
                details[:info] = error_info_elem.to_s
                # Also extract any child elements as separate fields
                error_info_elem.each_element do |child|
                  # Use local name without namespace prefix
                  field_name = "info_#{child.name.gsub(%r{^.*:}, '')}".to_sym
                  details[field_name] = child.text
                end
              end

              # Log the error with all available details (redacting sensitive data)
              logger.error("  Severity: #{details[:severity]}, Tag: #{details[:tag]}, Type: #{details[:type]}")
              if details[:message]
                safe_message = @redactor ? @redactor.redact_string(details[:message]) : details[:message]
                logger.error("  Message: #{safe_message}")
              end
              if details[:path]
                safe_path = @redactor ? @redactor.redact_string(details[:path]) : details[:path]
                logger.error("  Path: #{safe_path}")
              end
              logger.error("  App-tag: #{details[:app_tag]}") if details[:app_tag]
              if details[:info]
                safe_log_xml(logger, :error, '  Info:', details[:info])
              end

              details.compact # Remove nil values
            end

            # Check for severe errors - handle the first one
            severe_errors = error_details.select { |e| e[:severity] == 'error' }
            if severe_errors.any?
              # Handle the first error (most NETCONF operations fail fast on first error)
              first_error = severe_errors.first
              error_msg = first_error[:message] || 'NETCONF RPC Error'

              # Raise specific exception based on error tag (RFC 6241)
              error_class = case first_error[:tag]
                            # Resource/Lock related
                            when 'in-use' then NetconfError::ResourceInUse
                            when 'lock-denied' then NetconfError::LockDenied
                            when 'resource-denied' then NetconfError::ResourceDenied
                            # Value/Attribute errors
                            when 'invalid-value' then NetconfError::InvalidValue
                            when 'missing-attribute' then NetconfError::MissingAttribute
                            when 'bad-attribute' then NetconfError::BadAttribute
                            when 'unknown-attribute' then NetconfError::UnknownAttribute
                            # Element errors
                            when 'missing-element' then NetconfError::MissingElement
                            when 'bad-element' then NetconfError::BadElement
                            when 'unknown-element' then NetconfError::UnknownElement
                            when 'unknown-namespace' then NetconfError::UnknownNamespace
                            # Data errors
                            when 'data-exists' then NetconfError::DataExists
                            when 'data-missing' then NetconfError::DataMissing
                            # Operation errors
                            when 'operation-not-supported' then NetconfError::OperationNotSupported
                            when 'operation-failed' then NetconfError::OperationFailed
                            when 'partial-operation' then NetconfError::PartialOperation
                            when 'malformed-message' then NetconfError::MalformedMessage
                            # Other errors
                            when 'access-denied' then NetconfError::AccessDenied
                            when 'rollback-failed' then NetconfError::RollbackFailed
                            when 'too-big' then NetconfError::TooBig
                            # Default to generic RpcError for unknown tags
                            else NetconfError::RpcError
                            end

              # Pass complete error details to preserve all context
              raise error_class.new(error_msg, first_error)
            end
          end

          # Check for OK response
          has_ok = REXML::XPath.match(rsp_nx, '//nc:ok', 'nc' => NAMESPACE).any?
          if has_ok
            logger.debug('NETCONF RPC successful - received <ok/> response')
          end

          # For operations that must return <ok/>, validate it was received
          operation_name = cmd_nx.name
          operations_requiring_ok = ['edit-config', 'copy-config', 'delete-config', 'lock', 'unlock', 'commit', 'discard-changes', 'close-session', 'kill-session']

          if operations_requiring_ok.include?(operation_name) && !has_ok && rpc_errs.empty?
            logger.error("NETCONF operation '#{operation_name}' did not return expected <ok/> response")
            raise NetconfError::RpcError, "Operation '#{operation_name}' did not return expected <ok/> response"
          end

          # Return child elements (excluding text nodes)
          rsp_nx.elements.to_a
        end

        private

        def start_monitoring
          logger = get_logger
          logger&.debug("NETCONF starting monitoring thread (keepalive: #{@keepalive_interval}s, idle timeout: #{@options[:idle_timeout]}s)")

          @keepalive_thread = Thread.new do
            # Sleep for the minimum interval needed for checks
            check_interval = [@keepalive_interval, @options[:idle_timeout] || Float::INFINITY].compact.min

            loop do
              sleep(check_interval)

              # Check if we need to send a keepalive
              break unless @state == :NETCONF_OPEN
              time_since_last_activity = Time.now - @last_activity

              # Check for idle timeout first
              if @options[:idle_timeout] && time_since_last_activity >= @options[:idle_timeout]
                logger&.warn("NETCONF connection idle timeout exceeded (#{time_since_last_activity.round}s >= #{@options[:idle_timeout]}s)")
                logger&.info('NETCONF closing connection due to idle timeout')

                # Close the connection
                disconnect
                break
              elsif time_since_last_activity >= @keepalive_interval
                logger&.trace("NETCONF sending keepalive (idle for #{time_since_last_activity.round}s)")

                # Send an empty RPC as keepalive
                keepalive_doc = REXML::Document.new('<rpc/>')
                keepalive_rpc = keepalive_doc.root
                keepalive_rpc.add_namespace(NAMESPACE)
                keepalive_rpc.add_attribute('message-id', @rpc_message_id.to_s)
                @rpc_message_id += 1

                begin
                  send_and_receive(keepalive_doc.to_s)
                  logger&.trace('NETCONF keepalive sent successfully')
                rescue => e
                  logger&.warn("NETCONF keepalive failed: #{e.message}")
                  # Don't break the loop on keepalive failure
                end
              end

            # Connection closed, exit thread

            rescue => e
              logger&.error("NETCONF keepalive thread error: #{e.message}")
              break
            end

            logger&.debug('NETCONF keepalive thread exiting')
          end
        end

        def stop_monitoring
          return unless @keepalive_thread
          logger = get_logger
          logger&.debug('NETCONF stopping monitoring thread')
          @keepalive_thread.kill
          @keepalive_thread.join(1) # Wait up to 1 second for thread to finish
          @keepalive_thread = nil
        end

        def reconnectable_error?(error)
          # Determine if an error is recoverable through reconnection
          case error
          when Errno::ECONNRESET, Errno::EPIPE, Errno::ECONNABORTED,
               Net::SSH::Disconnect, Net::SSH::ConnectionTimeout,
               IOError
            true
          when NetconfError::RpcError
            # Timeout errors might be recoverable
            error.message.include?('timed out') || error.message.include?('timeout')
          else
            false
          end
        end

        def attempt_reconnect
          logger = get_logger

          # Stop monitoring thread before reconnecting
          stop_monitoring

          # Close existing connection
          begin
            transaction_close
          rescue => e
            logger&.debug("Error closing connection during reconnect: #{e.message}")
          end

          @state = :NETCONF_CLOSED

          # Wait before reconnecting
          sleep(@reconnect_delay)

          # Try to reconnect
          begin
            logger&.info("Attempting NETCONF reconnection to #{@host}")

            # Clear connection state
            @transaction = {}
            @rpc_message_id = 1

            # Reconnect
            raise NetconfError::OpenError unless transaction_open

            logger&.trace('NETCONF waiting for server hello (reconnect)')
            hello_xml = transaction_receive_hello
            logger&.trace('NETCONF server hello received (reconnect)')

            hello_rsp = REXML::Document.new(hello_xml)

            # Re-parse capabilities and session ID
            @capabilities = REXML::XPath.match(hello_rsp, '//nc:capability', 'nc' => NAMESPACE).map(&:text)
            @server_capabilities = @capabilities
            new_session_id = REXML::XPath.first(hello_rsp, '//nc:session-id', 'nc' => NAMESPACE).text

            logger&.info("NETCONF reconnected with new session-id: #{new_session_id} (was: #{@session_id})")
            @session_id = new_session_id

            # Send client hello
            transaction_send_hello

            @state = :NETCONF_OPEN

            # Restart monitoring if enabled
            start_monitoring if @keepalive_interval > 0 || @options[:idle_timeout]

            true
          rescue => e
            logger&.error("NETCONF reconnection failed: #{e.message}")
            @state = :NETCONF_CLOSED
            false
          end
        end

        # Safely log XML content, redacting sensitive data if a redactor is available
        def safe_log_xml(logger, level, message, xml_content)
          return unless logger

          # Map trace to debug if logger doesn't support trace
          actual_level = (level == :trace && !logger.respond_to?(:trace)) ? :debug : level

          # Send the message line
          logger.send(actual_level, message)

          # Redact and send the XML content
          sanitized_content = if @redactor&.respond_to?(:redact_xml)
                                begin
                                  @redactor.redact_xml(xml_content)
                                rescue => e
                                  # If redaction fails, don't log the content at all to prevent sensitive data exposure
                                  logger.error("Failed to redact XML for logging: #{e.message}")
                                  '[REDACTION FAILED - CONTENT HIDDEN FOR SECURITY]'
                                end
                              else
                                # No redactor available - log content as-is (assumes user explicitly disabled redaction)
                                xml_content
                              end

          logger.send(actual_level, sanitized_content)
        end

        def negotiate_protocol_version(server_capabilities)
          # If version is forced, use it (with validation)
          if @force_version
            case @force_version
            when '1.0'
              return '1.0'
            when '1.1'
              # Check if server supports 1.1 when forced
              return '1.1' if server_capabilities.any? { |cap| cap.include?('urn:ietf:params:netconf:base:1.1') }

              logger = get_logger
              logger&.warn("NETCONF 1.1 forced but server doesn't support it, falling back to 1.0")
              return '1.0'

            else
              raise ArgumentError, "Invalid force_version: #{@force_version}. Must be '1.0' or '1.1'"
            end
          end

          # Normal negotiation: use highest mutually supported version
          if server_capabilities.any? { |cap| cap.include?('urn:ietf:params:netconf:base:1.1') }
            '1.1'
          else
            '1.0'
          end
        end

        def get_logger
          # Use the Session's logger if available, otherwise create a default one
          logger = if defined?(PuppetX::Puppetlabs::Netconf::Session) && PuppetX::Puppetlabs::Netconf::Session.respond_to?(:logger)
                     PuppetX::Puppetlabs::Netconf::Session.logger
                   else
                     # Create a default logger for standalone use
                     @logger ||= begin
                       logger = Logger.new(STDERR)
                       logger.level = Logger::WARN
                       logger.progname = "NetconfClient[#{@host}]"
                       logger
                     end
                   end

          # Add trace method if it doesn't exist (for Ruby's Logger)
          unless logger.respond_to?(:trace)
            def logger.trace(msg)
              debug(msg) if level <= Logger::DEBUG
            end
          end

          logger
        end

        def validate_key_file(path)
          # Check if file exists
          unless File.exist?(path)
            raise NetconfError::ConfigurationError.new(
              "Private key file does not exist: #{path}. Please check the file path or use password authentication instead.",
              { key_path: path },
            )
          end

          # Check if file is readable
          unless File.readable?(path)
            raise NetconfError::ConfigurationError.new(
              "Private key file is not readable: #{path}. Please check file permissions (chmod 600 recommended for private keys).",
              { key_path: path },
            )
          end

          # Check if it's a regular file (not a directory)
          unless File.file?(path)
            raise NetconfError::ConfigurationError.new(
              "Private key path is not a file: #{path}. Please provide the path to the private key file, not a directory.",
              { key_path: path },
            )
          end

          # Basic check that it looks like a private key
          begin
            content = File.read(path, 100) # Read first 100 bytes
            unless content.match?(%r{-----BEGIN .* PRIVATE KEY-----}) || content.include?('-----BEGIN RSA PRIVATE KEY-----')
              logger = get_logger
              logger&.warn("File at #{path} may not be a valid private key file")
            end
          rescue => e
            # Don't fail if we can't read the file content for validation
            logger = get_logger
            logger&.debug("Could not validate private key file content: #{e.message}")
          end
        end

        def transaction_open
          start_args = {
            port: @options[:port] || NETCONF_PORT,
            timeout: @options[:connect_timeout],
            # NETCONF-specific SSH options matching working configurations
            keys_only: false,
            verify_host_key: :never,
            use_agent: false,
            non_interactive: true,
            number_of_password_prompts: 1
          }

          # Configure authentication based on available credentials
          if @options[:private_key]
            # Private key authentication takes precedence over password
            if @options[:password]
              logger = get_logger
              logger&.warn('Both private key and password provided. Using private key authentication.')
            end

            start_args[:auth_methods] = ['publickey']

            # Handle both file path and key data formats (like Bolt SSH transport)
            if @options[:private_key].is_a?(String)
              # It's a file path - validate it exists and is readable
              key_path = @options[:private_key]
              begin
                validate_key_file(key_path)
                start_args[:keys] = [key_path]
              rescue NetconfError::ConfigurationError => e
                logger = get_logger
                logger&.error("Private key validation failed: #{e.message}")
                raise
              end
            elsif @options[:private_key].is_a?(Hash) && @options[:private_key]['key-data']
              # It's key content from bolt-server
              key_data = @options[:private_key]['key-data']

              # Basic validation that it looks like a private key
              unless key_data.match?(%r{-----BEGIN (RSA |DSA |EC |OPENSSH )?PRIVATE KEY-----})
                raise NetconfError::ConfigurationError.new(
                  'Private key content does not appear to be a valid private key format',
                  { error_detail: 'Missing private key header' },
                )
              end

              start_args[:key_data] = [key_data]
            else
              # Unexpected format
              raise NetconfError::ConfigurationError.new(
                "Private key must be either a file path string or a hash with 'key-data'",
                { provided_type: @options[:private_key].class.name },
              )
            end
          else
            # Password authentication
            start_args[:auth_methods] = ['password', 'keyboard-interactive']
            start_args[:password] = @options[:password] if @options[:password]
          end

          start_args.merge!(@options[:ssh_args]) if @options[:ssh_args]

          begin
            @transaction[:conn] = Net::SSH.start(@host, @username, start_args)
            @transaction[:chan] = @transaction[:conn].open_channel do |ch|
              ch.subsystem(NETCONF_SUBSYSTEM)
            end
          rescue Errno::ECONNREFUSED => e
            logger = get_logger
            logger&.error("Connection refused to #{@host}:#{start_args[:port]} - #{e.message}")
            raise NetconfError::OpenError.new(
              "Connection refused to #{@host}:#{start_args[:port]}: #{e.message}",
              { host: @host, port: start_args[:port] },
            )
          rescue Net::SSH::AuthenticationFailed => e
            logger = get_logger
            logger&.error("Authentication failed for #{@username}@#{@host} - #{e.message}")
            raise NetconfError::AuthenticationError.new(
              "Authentication failed for #{@username}@#{@host}: #{e.message}",
              { host: @host, username: @username },
            )
          rescue Net::SSH::HostKeyError => e
            logger = get_logger
            logger&.error("Host key verification failed for #{@host} - #{e.message}")
            raise NetconfError::HostKeyError.new(
              "Host key verification failed for #{@host}: #{e.message}",
              { host: @host, fingerprint: begin
                                            e.fingerprint
                                          rescue
                                            nil
                                          end },
            )
          rescue Net::SSH::ConnectionTimeout
            timeout_value = start_args[:timeout] || 10
            logger = get_logger
            logger&.error("Connection timeout after #{timeout_value}s to #{@host}:#{start_args[:port]}")
            raise NetconfError::TimeoutError.new(
              "Timeout after #{timeout_value} seconds connecting to #{@host}:#{start_args[:port]}",
              :connect,
              timeout_value,
            )
          rescue Net::SSH::Timeout => e
            # Generic SSH timeout (different from ConnectionTimeout)
            logger = get_logger
            logger&.error("SSH operation timeout for #{@host} - #{e.message}")
            raise NetconfError::TimeoutError.new(
              "SSH operation timed out for #{@host}: #{e.message}",
              :ssh_operation,
              nil,
            )
          rescue => e
            logger = get_logger
            logger&.error("Unexpected error connecting to #{@host}: #{e.class} - #{e.message}")
            raise NetconfError::OpenError.new(
              "Failed to connect to #{@host}: #{e.message}",
              { host: @host, error_class: e.class.name },
            )
          end
          @transaction[:chan]
        end

        def transaction_close
          @transaction[:chan]&.close
          @transaction[:conn]&.close
        end

        def transaction_receive
          logger = get_logger
          logger.trace('NETCONF transaction_receive starting')

          @transaction[:rx_buf] = +''
          @transaction[:more] = true
          bytes_received = 0

          if @protocol_version == '1.1'
            # For NETCONF 1.1, we need to handle chunked framing
            @transaction[:chunk_buffer] = +''
            @transaction[:in_chunk] = false
            @transaction[:expected_chunk_size] = 0

            @transaction[:chan].on_data do |_ch, data|
              bytes_received += data.bytesize
              logger.trace("NETCONF 1.1 received #{data.bytesize} bytes, total: #{bytes_received}")

              @transaction[:chunk_buffer] << data

              # Check if we have a complete chunked message
              if @transaction[:chunk_buffer].include?(CHUNK_END)
                logger.trace('NETCONF 1.1 received end-of-chunks marker')
                @transaction[:more] = false
              end
            end
          else
            # For NETCONF 1.0, use the EOM delimiter
            @transaction[:chan].on_data do |_ch, data|
              bytes_received += data.bytesize
              logger.trace("NETCONF received #{data.bytesize} bytes, total: #{bytes_received}")

              if data.include?(MSG_END)
                logger.trace('NETCONF received MSG_END marker')
                data.slice!(MSG_END)
                @transaction[:rx_buf] << data unless data.empty?
                @transaction[:more] = false
              else
                @transaction[:rx_buf] << data
              end
            end
          end

          @transaction[:chan].on_extended_data do |_ch, _type, data|
            logger.error("NETCONF received extended data (error): #{data}")
            @transaction[:rx_err] = data
            @transaction[:more] = false
          end

          # Add timeout to prevent hanging
          timeout = @options[:command_timeout]
          begin
            require 'timeout'
            Timeout.timeout(timeout) do
              @transaction[:conn].loop { @transaction[:more] }
            end
          rescue Timeout::Error
            logger.error("NETCONF receive timeout after #{timeout} seconds, received #{bytes_received} bytes so far")
            @transaction[:more] = false
            if @transaction[:rx_buf].empty?
              # Connection may be in inconsistent state after timeout
              @state = :NETCONF_CLOSED
              raise NetconfError::TimeoutError.new(
                "NETCONF operation timed out after #{timeout} seconds - connection closed",
                :command_timeout,
                timeout,
              )
            else
              logger.warn('NETCONF timeout but data was received, returning partial data')
            end
          end

          logger.trace("NETCONF transaction_receive completed, total bytes: #{@transaction[:rx_buf].bytesize}")

          # Decode chunked message if using NETCONF 1.1
          if @protocol_version == '1.1' && @transaction[:chunk_buffer]
            begin
              decoded_msg = decode_chunked(@transaction[:chunk_buffer])
              logger.trace("NETCONF 1.1 decoded message size: #{decoded_msg.bytesize}")
              decoded_msg
            rescue => e
              logger.error("NETCONF 1.1 chunked decoding failed: #{e.message}")
              raise
            end
          else
            @transaction[:rx_buf]
          end
        end

        def transaction_send(cmd_str)
          @transaction[:chan].send_data(cmd_str)
        end

        def transaction_receive_hello
          # Hello messages always use NETCONF 1.0 framing (EOM delimiter)
          # Save current protocol version and temporarily use 1.0
          saved_version = @protocol_version
          @protocol_version = '1.0'

          # Use hello timeout for initial handshake
          saved_timeout = @options[:command_timeout]
          @options[:command_timeout] = @options[:hello_timeout]

          begin
            result = transaction_receive
          ensure
            # Restore protocol version and timeout
            @protocol_version = saved_version
            @options[:command_timeout] = saved_timeout
          end

          result
        end

        def transaction_send_hello
          transaction_send(MSG_HELLO)
          transaction_send(MSG_END)
        end

        # Encode message using NETCONF 1.1 chunked framing
        def encode_chunked(data)
          encoded = +'' # Use unfrozen string

          # Split data into chunks
          offset = 0
          while offset < data.bytesize
            chunk_data = data.byteslice(offset, CHUNK_SIZE)
            chunk_size = chunk_data.bytesize

            # Add chunk header and data
            encoded << "\n##{chunk_size}\n#{chunk_data}"
            offset += chunk_size
          end

          # Add end-of-chunks marker
          encoded << CHUNK_END
          encoded
        end

        # Decode NETCONF 1.1 chunked message
        def decode_chunked(data)
          logger = get_logger
          decoded = +'' # Use unfrozen string
          buffer = data.dup

          until buffer.empty?
            # Check for end-of-chunks marker first
            if buffer.start_with?(CHUNK_END)
              buffer.slice!(0, CHUNK_END.length)
              break
            end

            # Look for chunk header
            if (match = buffer.match(%r{\A\n#(\d+)\n}))
              chunk_size = match[1].to_i
              header_length = match[0].length

              # Extract chunk data
              if buffer.length >= header_length + chunk_size
                buffer.slice!(0, header_length) # Remove header
                chunk_data = buffer.slice!(0, chunk_size)
                decoded << chunk_data
              else
                logger&.error('NETCONF chunked framing error: incomplete chunk data')
                raise NetconfError::RpcError, 'Invalid chunked framing: incomplete chunk'
              end
            else
              logger&.error('NETCONF chunked framing error: invalid chunk header')
              raise NetconfError::RpcError, 'Invalid chunked framing: expected chunk header'
            end
          end

          decoded
        end

        # Internal class that executes NETCONF RPC operations.
        # @api private
        class RpcExecutor
          def initialize(client)
            @client = client
          end

          def method_missing(method, *args, &block)
            rpc_name = method.to_s.tr('_', '-')

            case rpc_name
            when 'get-config'
              get_config(*args, &block)
            when 'edit-config'
              edit_config(*args, &block)
            when 'get'
              get(*args, &block)
            when 'commit'
              commit
            when 'lock'
              lock(*args)
            when 'unlock'
              unlock(*args)
            when 'validate'
              validate(*args)
            when 'discard-changes'
              discard_changes
            when 'copy-config'
              copy_config(*args)
            when 'delete-config'
              delete_config(*args)
            when 'get-capabilities'
              get_capabilities
            when 'get-schema'
              get_schema(*args)
            when 'kill-session'
              kill_session(*args)
            else
              build_and_execute_rpc(rpc_name, *args)
            end
          end

          def respond_to_missing?(_method, _include_private = false)
            true
          end

          private

          def get_config(source = 'running', filter = nil, &block)
            logger = get_logger
            logger.debug("RpcExecutor get_config called: source=#{source}, filter=#{filter.nil? ? 'none' : 'provided'}")

            rpc_doc = REXML::Document.new("<rpc><get-config><source><#{source}/></source></get-config></rpc>")
            rpc = rpc_doc.root

            if block
              # For block-based filters, we need to build XML differently with REXML
              # This is a complex case that would need refactoring
              raise NotImplementedError, 'Block-based filters not yet supported with REXML'
            elsif filter
              get_config_elem = rpc.elements['get-config']
              f_node = REXML::Element.new('filter')
              f_node.add_attribute('type', 'subtree')
              if filter.is_a?(String)
                # Parse the string as XML and add the root element
                filter_doc = REXML::Document.new(filter)
                f_node.add(filter_doc.root.dup)
              else
                f_node.add(filter.dup)
              end
              get_config_elem.add(f_node)
            end

            @client.send(:safe_log_xml, logger, :trace, 'RpcExecutor get_config complete RPC:', rpc_doc)

            result = @client.rpc_exec(rpc)
            logger.debug("RpcExecutor get_config result class: #{result.class}")
            result
          end

          def edit_config(target = 'candidate', config = nil, default_operation = 'merge', &block)
            logger = get_logger
            logger.debug("RpcExecutor edit_config called: target=#{target}, default_operation=#{default_operation}")

            rpc_str = <<-EO_RPC.gsub(%r{^\s*\|}, '')
              |<rpc>
              |  <edit-config>
              |     <target><#{target}/></target>
              |     <config/>
              |  </edit-config>
              |</rpc>
            EO_RPC

            rpc_doc = REXML::Document.new(rpc_str)
            rpc = rpc_doc.root

            if block
              # For block-based config, we need to build XML differently with REXML
              raise NotImplementedError, 'Block-based config not yet supported with REXML'
            elsif config
              config_elem = rpc.elements['edit-config/config']
              if config.is_a?(String)
                # Parse the XML string and add its content to the config element
                begin
                  config_doc = REXML::Document.new(config)
                  # If the root element is <config>, use its children
                  if config_doc.root && config_doc.root.name == 'config'
                    config_doc.root.elements.each do |elem|
                      config_elem.add(elem.dup)
                    end
                  # Otherwise add the root element itself
                  elsif config_doc.root
                    config_elem.add(config_doc.root.dup)
                  end
                rescue REXML::ParseException => e
                  logger.error("Failed to parse config XML: #{e.message}")
                  raise ArgumentError, "Invalid XML in config: #{e.message}"
                end
              else
                config_elem.add(config.dup)
              end
            else
              raise ArgumentError, 'You must specify edit-config data!'
            end

            # Log the complete RPC before sending (with redaction)
            @client.send(:safe_log_xml, logger, :trace, 'RpcExecutor edit_config complete RPC:', rpc_doc)

            result = @client.rpc_exec(rpc)
            logger.debug("RpcExecutor edit_config result class: #{result.class}")
            result
          end

          def get(filter = nil, &block)
            logger = get_logger
            rpc_doc = REXML::Document.new('<rpc><get/></rpc>')
            rpc = rpc_doc.root

            if block
              logger.debug('RpcExecutor get called with block')
              # For block-based filters, we need to build XML differently with REXML
              raise NotImplementedError, 'Block-based filters not yet supported with REXML'
            elsif filter
              logger.debug('RpcExecutor get called with filter element')
              get_elem = rpc.elements['get']
              f_node = REXML::Element.new('filter')
              f_node.add_attribute('type', 'subtree')
              if filter.is_a?(String)
                # Parse the string as XML and add the root element
                filter_doc = REXML::Document.new(filter)
                f_node.add(filter_doc.root.dup)
              else
                f_node.add(filter.dup)
              end
              get_elem.add(f_node)
            else
              logger.debug('RpcExecutor get called without filter')
            end

            @client.rpc_exec(rpc)
          end

          def commit
            logger = get_logger
            logger.debug('RpcExecutor commit called')

            rpc_doc = REXML::Document.new('<rpc><commit/></rpc>')
            rpc = rpc_doc.root

            @client.send(:safe_log_xml, logger, :trace, 'RpcExecutor commit RPC:', rpc_doc)

            result = @client.rpc_exec(rpc)
            logger.debug("RpcExecutor commit result class: #{result.class}")
            result
          end

          def lock(target)
            logger = get_logger
            logger.debug("RpcExecutor lock called: target=#{target}")

            rpc_doc = REXML::Document.new("<rpc><lock><target><#{target}/></target></lock></rpc>")
            rpc = rpc_doc.root

            @client.send(:safe_log_xml, logger, :trace, 'RpcExecutor lock RPC:', rpc_doc)

            result = @client.rpc_exec(rpc)
            logger.debug("RpcExecutor lock result class: #{result.class}")
            result
          end

          def unlock(target)
            logger = get_logger
            logger.debug("RpcExecutor unlock called: target=#{target}")

            rpc_doc = REXML::Document.new("<rpc><unlock><target><#{target}/></target></unlock></rpc>")
            rpc = rpc_doc.root

            @client.send(:safe_log_xml, logger, :trace, 'RpcExecutor unlock RPC:', rpc_doc)

            result = @client.rpc_exec(rpc)
            logger.debug("RpcExecutor unlock result class: #{result.class}")
            result
          end

          def validate(source)
            rpc_doc = REXML::Document.new("<rpc><validate><source><#{source}/></source></validate></rpc>")
            rpc = rpc_doc.root
            @client.rpc_exec(rpc)
          end

          def discard_changes
            rpc_doc = REXML::Document.new('<rpc><discard-changes/></rpc>')
            rpc = rpc_doc.root
            @client.rpc_exec(rpc)
          end

          def kill_session(session_id)
            rpc_doc = REXML::Document.new("<rpc><kill-session><session-id>#{session_id}</session-id></kill-session></rpc>")
            rpc = rpc_doc.root
            @client.rpc_exec(rpc)
          end

          def copy_config(source, target)
            rpc_doc = REXML::Document.new("<rpc><copy-config><source><#{source}/></source><target><#{target}/></target></copy-config></rpc>")
            rpc = rpc_doc.root
            @client.rpc_exec(rpc)
          end

          def delete_config(target)
            rpc_doc = REXML::Document.new("<rpc><delete-config><target><#{target}/></target></delete-config></rpc>")
            rpc = rpc_doc.root
            @client.rpc_exec(rpc)
          end

          def get_capabilities
            { 'capabilities' => @client.server_capabilities }
          end

          def get_schema(identifier, version = nil, format = 'yang')
            logger = get_logger
            logger.debug("RpcExecutor get_schema called: identifier=#{identifier}, version=#{version}, format=#{format}")

            # Build get-schema RPC with proper namespace
            rpc_doc = REXML::Document.new
            rpc = REXML::Element.new('rpc')
            rpc_doc.add(rpc)
            get_schema = REXML::Element.new('get-schema')
            get_schema.add_namespace('urn:ietf:params:xml:ns:yang:ietf-netconf-monitoring')
            identifier_elem = REXML::Element.new('identifier')
            identifier_elem.add_text(identifier)
            get_schema.add(identifier_elem)
            if version
              version_elem = REXML::Element.new('version')
              version_elem.add_text(version)
              get_schema.add(version_elem)
            end
            format_elem = REXML::Element.new('format')
            format_elem.add_text(format)
            get_schema.add(format_elem)
            rpc.add(get_schema)

            @client.send(:safe_log_xml, logger, :trace, 'RpcExecutor get_schema complete RPC:', rpc_doc)

            @client.rpc_exec(rpc)
          end

          def build_and_execute_rpc(rpc_name, params = nil)
            rpc_doc = REXML::Document.new
            rpc = REXML::Element.new('rpc')
            rpc_doc.add(rpc)

            if params
              op_elem = REXML::Element.new(rpc_name)
              params.each do |k, v|
                param_name = k.to_s.tr('_', '-')
                param_elem = REXML::Element.new(param_name)
                param_elem.add_text(v.to_s) unless v == true
                op_elem.add(param_elem)
              end
              rpc.add(op_elem)
            else
              rpc.add(REXML::Element.new(rpc_name))
            end

            @client.rpc_exec(rpc)
          end

          def get_logger
            @client.send(:get_logger)
          end
        end
      end
    end
  end
end
