# frozen_string_literal: true

require 'rexml/document'

# Redactor provides functionality to redact sensitive data from NETCONF responses
# and configuration data. It works with XML data and plain text strings.
#
# The redactor identifies sensitive data by matching XML element names
# against configured patterns. When a match is found, the value
# is replaced with '[REDACTED]'.
#
# @example Basic usage
#   redactor = Redactor.new(['password', 'secret', 'key'])
#   redacted_xml = redactor.redact_xml(xml_string)
#   redacted_str = redactor.redact_string(error_message)
#
# @since 1.0.0
class PuppetX::Puppetlabs::Netconf::Redactor
  # Default patterns that commonly contain sensitive data
  DEFAULT_PATTERNS = [
    'password',
    'passwd',
    'secret',
    'key',
    'token',
    'credential',
    'auth',
    'community',
    'private',
    'encrypted',
  ].freeze

  # Placeholder text for redacted content
  REDACTED_TEXT = '[REDACTED]'

  # Initialize a new Redactor instance
  #
  # @param patterns [Array<String>, nil] Element/key names to redact.
  #   If nil, uses DEFAULT_PATTERNS. Patterns are case-insensitive.
  # @param include_defaults [Boolean] Whether to include default patterns
  #   in addition to custom patterns (default: false)
  def initialize(patterns = nil, include_defaults: false)
    @patterns = if patterns && include_defaults
                  # Merge custom patterns with defaults
                  (patterns + DEFAULT_PATTERNS).uniq
                else
                  patterns || DEFAULT_PATTERNS
                end
    # Convert patterns to lowercase for case-insensitive matching
    @patterns = @patterns.map(&:downcase)
  end

  # Redact sensitive data from XML content
  #
  # This method parses XML and replaces the text content of elements
  # whose names match the configured patterns with '[REDACTED]'.
  # It preserves XML structure and attributes.
  #
  # @param xml_content [String, REXML::Document, REXML::Element, Array<REXML::Element>]
  #   The XML content to redact. Can be a string, document, element, or array of elements.
  # @return [String] The redacted XML as a string
  def redact_xml(xml_content)
    return '' if xml_content.nil? || xml_content == ''

    # Handle different input types
    case xml_content
    when String
      # Parse the XML string
      begin
        # Parse and immediately create a new document to ensure it's not frozen
        source_doc = REXML::Document.new(xml_content)
        if source_doc.root
          # Create a completely new document with a deep clone
          new_doc = REXML::Document.new
          new_root = source_doc.root.deep_clone
          new_doc.add_element(new_root)
          redact_xml_element(new_root)
          new_doc.to_s
        else
          xml_content
        end
      rescue REXML::ParseException
        # If XML parsing fails, try basic string redaction
        redact_string(xml_content)
      end
    when Array
      # Handle array of REXML elements (common from NETCONF responses)
      # Create a new document to hold the redacted elements
      result_doc = REXML::Document.new
      result_doc.add_element(REXML::Element.new('root'))

      xml_content.each do |element|
        next unless element.is_a?(REXML::Element)
        # Deep clone to avoid modifying frozen elements
        element_copy = element.deep_clone
        redact_xml_element(element_copy)
        result_doc.root.add_element(element_copy)
      end

      # Return just the inner content without the wrapper root
      result_doc.root.elements.map(&:to_s).join
    when REXML::Document
      doc_copy = xml_content.deep_clone
      redact_xml_element(doc_copy.root) if doc_copy.root
      doc_copy.to_s
    when REXML::Element
      element_copy = xml_content.deep_clone
      redact_xml_element(element_copy)
      element_copy.to_s
    else
      # Log warning for unexpected content type and convert to string
      warn "Redactor#redact_xml received unexpected content type: #{xml_content.class}. Converting to string."
      xml_content.to_s
    end
  end

  # Redact sensitive data from a plain string
  #
  # This method is used for error messages or other string content
  # that might contain sensitive data but isn't structured XML.
  #
  # @param str [String] The string to redact
  # @return [String] The redacted string
  def redact_string(str)
    return str unless str.is_a?(String)

    result = str.dup

    # Look for common patterns in the string
    # This is more aggressive and looks for patterns like:
    # password="value", password: value, <password>value</password>
    @patterns.each do |pattern|
      # XML element pattern: <pattern>content</pattern>
      result.gsub!(%r{<#{Regexp.escape(pattern)}>([^<]*)</#{Regexp.escape(pattern)}>}i,
                  "<#{pattern}>#{REDACTED_TEXT}</#{pattern}>")

      # XML attribute pattern: pattern="value"
      result.gsub!(%r{#{Regexp.escape(pattern)}="[^"]*"}i,
                  "#{pattern}=\"#{REDACTED_TEXT}\"")

      # Ruby hash pattern: pattern: value or pattern => value
      # Simplified to avoid complex character class
      result.gsub!(%r{#{Regexp.escape(pattern)}(?:\s*=>?\s*|:\s*)["']?[^"',\s\}]+["']?}i,
                  "#{pattern}: #{REDACTED_TEXT}")
    end

    result
  end

  private

  # Recursively redact sensitive elements in an XML element tree
  #
  # @param element [REXML::Element] The element to process
  def redact_xml_element(element)
    return unless element.is_a?(REXML::Element)

    # Check if this element's name matches a pattern
    if should_redact?(element.name)
      # Replace text content with redacted placeholder
      if element.has_text? && !element.text.to_s.strip.empty?
        # Remove all text nodes by iterating backwards to avoid index issues
        element.children.size.downto(0) do |i|
          child = element.children[i]
          if child&.is_a?(REXML::Text)
            element.delete(child)
          end
        end
        # Add the redacted text
        element.add_text(REDACTED_TEXT)
      end

      # Also redact any attributes that match patterns
      element.attributes.each do |name, _attr|
        if should_redact?(name)
          element.attributes[name] = REDACTED_TEXT
        end
      end
    end

    # Also check attributes even if element name doesn't match
    element.attributes.each do |name, _attr|
      if should_redact?(name)
        element.attributes[name] = REDACTED_TEXT
      end
    end

    # Recursively process child elements
    element.elements.each do |child|
      redact_xml_element(child)
    end
  end

  # Check if a name should be redacted based on patterns
  #
  # @param name [String] The element or attribute name to check
  # @return [Boolean] true if the name matches a redaction pattern
  def should_redact?(name)
    return false if name.nil? || name.empty?

    name_lower = name.downcase

    # Check for exact matches
    return true if @patterns.include?(name_lower)

    # Check if the name contains any of the patterns
    # This handles cases like 'password-hashed', 'ssh-key', 'auth-token', etc.
    @patterns.any? do |pattern|
      name_lower.include?(pattern)
    end
  end
end
