#!/usr/bin/env ruby
# frozen_string_literal: true

# @!visibility private
# @summary Retrieves configuration from a NETCONF device
#
# This task demonstrates the basic NETCONF get-config operation, allowing you to
# retrieve configuration data from any supported datastore on the target device.
#
# @example Get full running configuration
#   bolt task run netconf::get_config --targets router1
#
# @example Get configuration from candidate datastore
#   bolt task run netconf::get_config source=candidate --targets router1
#
# @example Get filtered configuration (interfaces only)
#   bolt task run netconf::get_config filter='<interfaces xmlns="urn:ietf:params:xml:ns:yang:ietf-interfaces"/>' --targets router1
#
# @param source [String] Configuration datastore to retrieve ('running', 'candidate', or 'startup')
#   Defaults to 'running'. The device must support the specified datastore.
#
# @param filter [String] Optional subtree or XPath filter to limit the response
#   Can be a complete XML subtree or XPath expression to retrieve specific configuration sections.
#
# @param source_datastore [String] Override the default source datastore for the session
#   Allows specifying a different default datastore for all operations in the session.
#
# @param redact_patterns [Array<String>] Patterns to redact from logs and output
#   Regular expressions that match sensitive data to be redacted.
#
# @return [Hash] Task result containing:
#   - 'config' [String]: The retrieved configuration as XML
#   - 'source' [String]: The datastore that was queried
#   - 'filtered' [Boolean]: Whether a filter was applied
#   - 'session_report' [Hash]: Detailed session information

require 'json'

# Include the task helper to set up load paths
require_relative '../files/task_helper.rb'

# Now require the netconf library
require 'puppet_x/puppetlabs/netconf/session'

result = PuppetX::Puppetlabs::Netconf::Session.with_session do |session|
  # Get task parameters
  source = session.task_params['source'] || session.source_datastore
  filter = session.task_params['filter']

  session.logger.info("Retrieving #{source} configuration from device")
  session.logger.debug("Filter provided: #{filter ? 'yes' : 'no'}")

  # Get the configuration
  if filter
    # Apply subtree filter if provided
    session.logger.debug('Applying subtree filter')
    config = session.get_config(filter)
  else
    # Get full configuration
    session.logger.debug('Getting full configuration')
    config = session.get_config
  end

  session.logger.info('Successfully retrieved configuration')

  # Convert the REXML elements to a string for readability
  config_text = config.map(&:to_s).join("\n")

  # Return the configuration via report_result
  session.report_result({
                          'config' => config_text,
    'datastore' => source,
    'filtered' => !filter.nil?
                        })
end

# Output the result as JSON for Bolt
puts result.to_json
