#!/usr/bin/env ruby
# frozen_string_literal: true

require 'json'
require_relative '../files/task_helper.rb'
require 'puppet_x/puppetlabs/netconf/session'

# Intelligently recover from lock issues
#
# This task analyzes lock problems and takes the appropriate action:
# - For session ID 0 locks: discards candidate changes
# - For stale session locks: kills the blocking session
# - For your own locks: unlocks them
# - Provides clear guidance on what happened and why

result = PuppetX::Puppetlabs::Netconf::Session.with_session do |session|
  # Smart default: use candidate if supported, otherwise running
  datastore = session.task_params['datastore']
  if datastore.nil?
    datastore = session.supports_candidate? ? 'candidate' : 'running'
  end
  force = session.task_params['force'] || false

  # Try to acquire the lock to see what's blocking us
  lock_info = {}

  begin
    # Attempt to lock the datastore
    session.lock(datastore)
    session.unlock(datastore)

    lock_info = {
      status: 'success',
      message: "Datastore #{datastore} is not locked - no recovery needed",
      action_taken: 'none',
      datastore: datastore
    }
  rescue PuppetX::Puppetlabs::Netconf::NetconfError::LockDenied => e
    # Parse the error to understand who holds the lock
    lock_holder = e[:session_id] || e[:'session-id'] || e[:info_session_id] || e[:'info_session-id']

    if lock_holder == '0'
      # Session ID 0 means non-NETCONF entity (CLI, web UI, or uncommitted changes)
      if datastore == 'candidate' && session.supports_candidate?
        begin
          session.discard_changes
          lock_info = {
            status: 'success',
            message: 'Recovered from session ID 0 lock by discarding candidate changes',
            action_taken: 'discard_changes',
            lock_holder: 'non-NETCONF entity (CLI/web/uncommitted changes)',
            datastore: datastore
          }
        rescue => discard_error
          lock_info = {
            status: 'error',
            message: "Session ID 0 lock detected but discard failed: #{discard_error.message}",
            action_taken: 'discard_changes_failed',
            lock_holder: 'non-NETCONF entity',
            datastore: datastore,
            recommendation: 'May need to clear lock via device CLI or wait for timeout'
          }
        end
      else
        lock_info = {
          status: 'error',
          message: 'Session ID 0 lock on non-candidate datastore cannot be auto-recovered',
          action_taken: 'none',
          lock_holder: 'non-NETCONF entity',
          datastore: datastore,
          recommendation: 'Clear lock via device CLI or wait for timeout'
        }
      end
    elsif lock_holder
      # Another NETCONF session holds the lock
      if force
        begin
          # Kill the session holding the lock
          session.kill_session(lock_holder)

          # Try to acquire lock to verify it worked
          session.lock(datastore)
          session.unlock(datastore)

          lock_info = {
            status: 'success',
            message: "Recovered by killing session #{lock_holder}",
            action_taken: 'kill_session',
            killed_session: lock_holder,
            datastore: datastore,
            warning: 'Any uncommitted changes from the killed session were preserved'
          }
        rescue => kill_error
          lock_info = {
            status: 'error',
            message: "Failed to kill blocking session: #{kill_error.message}",
            action_taken: 'kill_session_failed',
            lock_holder: lock_holder,
            datastore: datastore,
            recommendation: 'Check if you have permission to kill other sessions'
          }
        end
      else
        lock_info = {
          status: 'info',
          message: "Datastore locked by session #{lock_holder}",
          action_taken: 'none',
          lock_holder: lock_holder,
          datastore: datastore,
          recommendation: "Use 'force: true' to kill the blocking session, or wait for it to complete"
        }
      end
    else
      # Could not determine lock holder
      lock_info = {
        status: 'error',
        message: 'Lock denied but could not determine lock holder',
        action_taken: 'none',
        datastore: datastore,
        error_details: e.message,
        recommendation: 'Check device logs or try device-specific recovery'
      }
    end
  rescue => e
    lock_info = {
      status: 'error',
      message: "Unexpected error during lock recovery: #{e.message}",
      action_taken: 'none',
      datastore: datastore
    }
  end

  session.report_result(lock_info)
end

puts result.to_json
