# Coordinates inclusion and ordering of puppet_enterprise profile classes.
class pe_install::install {
  # PRIMARY/ORCHESTRATOR
  if $::pe_install::is_master {
    # Packages must be installed before profiles begin configuring anything.
    Class['Puppet_enterprise::Packages'] -> [
      Class['Puppet_enterprise::Profile::Master'],
    ]

    if !pe_compile_master() {
      # PE-15846: pe-client-tools needs to be installed first if it is in the catalog
      Package['pe-client-tools'] -> Package['pe-r10k']
      unless $::pe_install::is_upgrade {
        file { 'default site.pp':
          ensure  => file,
          path    => '/etc/puppetlabs/code/environments/production/manifests/site.pp',
          owner   => 'pe-puppet',
          group   => 'pe-puppet',
          mode    => '0640',
          replace => false,
          source  => 'puppet:///modules/pe_install/site.pp',
        }
      }
    }

    if !$::pe_install::is_database {
      include puppet_enterprise::postgresql::client
    }

    class { 'puppet_enterprise::profile::master': }
    class { 'pe_install::install::ssldir':
      # The pe_install::prepare::certificates class uses the `puppetserver ca`
      # commands to generate files as root; must wait for that to complete before
      # executing pe_install::install::ssldir
      require => [Package['pe-puppetserver'],Class['Pe_install::Prepare::Certificates']],
    }
    contain '::pe_install::install::ssldir'

    if $pe_install::is_orchestrator {
      Class['Puppet_enterprise::Packages'] -> [
        Class['Puppet_enterprise::Profile::Orchestrator'],
      ]
      class { 'puppet_enterprise::profile::orchestrator': }
    }

    # pe_repo sets the master parameter to $settings::server by default.
    # However on initial installs, the server setting isn't configured until the
    # catalog is getting applied, so it defaults to puppet.
    # If the user didn't specify an override, default to the
    # primary's certname instead for install.
    $_pe_repo_master = lookup('::pe_repo::master', { 'default_value' => $::pe_install::master_certname } )

    class { 'pe_repo':
      master => $_pe_repo_master
    }

    $_pe_repo_platform = platform_tag_to_pe_repo_class($facts['platform_tag'])
    contain "::pe_repo::platform::${_pe_repo_platform}"
    contain '::pe_repo::platform::windows_x86_64'
  }

  # CA
  if $::pe_install::is_ca {
    # Packages must be installed before profiles begin configuring anything.
    Class['Puppet_enterprise::Packages'] -> [
      Class['Puppet_enterprise::Profile::Certificate_authority'],
    ]
    class { 'puppet_enterprise::profile::certificate_authority': }
    contain 'pe_install::upgrade::migrate_ca_directory'
  }

  # DATABASE
  if $::pe_install::is_database {
    class { 'puppet_enterprise::profile::database': }
    Package['postgresql-server'] -> Class['puppet_enterprise::profile::database']
    if $::pe_install::is_puppetdb {
      Class['puppet_enterprise::profile::database'] -> Package['pe-puppetdb']
    }
    if $::pe_install::is_console {
      Class['puppet_enterprise::profile::database'] -> Package['pe-console-services']
    }
    if $::pe_install::is_master {
      Class['puppet_enterprise::profile::database'] -> Package['pe-orchestration-services']
    }
  }

  # PUPPETDB
  if $::pe_install::is_puppetdb {
    # Packages must be installed before profiles begin configuring anything.
    Class['Puppet_enterprise::Packages'] -> [
      Class['Puppet_enterprise::Profile::Puppetdb'],
    ]
    class { puppet_enterprise::profile::puppetdb: }
    contain '::pe_install::upgrade::puppetdb'
    if !$::pe_install::is_database {
      include puppet_enterprise::postgresql::client
    }
  }

  # CONSOLE
  if $::pe_install::is_console {
    # Packages must be installed before profiles begin configuring anything.
    Class['Puppet_enterprise::Packages'] -> [
      Class['Puppet_enterprise::Profile::Console'],
    ]
    if !$::pe_install::is_database {
      include puppet_enterprise::postgresql::client
    }

    class { 'puppet_enterprise::profile::console': }
    -> class { 'pe_install::install::classification': }

    if $::pe_install::console_admin_password {
      $_console_admin_password_lockfile = '/opt/puppetlabs/server/data/console_admin_password_lock'
      $_escaped_console_password = pe_shell_escape($::pe_install::console_admin_password)
      $_command = @("CMD"/L)
        /opt/puppetlabs/bin/puppet-infra console_password \
        --password ${_escaped_console_password} && \
        touch ${_console_admin_password_lockfile} && \
        chmod 400 ${_console_admin_password_lockfile}
        | -CMD
      exec { 'set console admin password':
        command => $_command,
        cwd     => '/opt/puppetlabs/server/bin',
        creates => $_console_admin_password_lockfile,
        require => [
          Service['pe-console-services'],
        ],
      }
    }
  }

  # AGENT
  Class['Puppet_enterprise::Packages'] -> [
    Class['Puppet_enterprise::Profile::Agent'],
    Class['Pe_infrastructure::puppet_infra_shims'],
  ]

  class { 'puppet_enterprise::profile::agent': }

  class { 'pe_infrastructure::puppet_infra_shims': }

  if $pe_install::puppet_service_managed == true {
    service { 'puppet':
      ensure => running,
      enable => true,
    }
  }
}
