require 'puppet/agent'
require 'puppet/indirector/face'
require 'puppet/error'
require 'puppet/util/pe_conf'
require 'hocon'

Puppet::Face.define(:infrastructure, '1.0.0') do
  extend Puppet::Agent::Locker

  action :recover_configuration do
    summary _('Recovers configuration set outside of pe.conf')

    description <<-EOT
      Generates a pe.conf and node specific configuration files with
      any PE configuration settings a user has specified through the
      PE Classifier or Hiera. Will exit (17) if a puppet run is
      already in progress.
    EOT

    option('--target-dir DIR') do
      default_to { Puppet::Util::Pe_conf::CONF_DIR }
      summary _("Directory to output config files to. (NOTE: the only supported path for puppet-infrastructure hiera configuration is #{Puppet::Util::Pe_conf::CONF_DIR}).")
    end

    option('--pe-environmentpath PATH') do
      summary _('The environmentpath to recover configuration from.')
      description <<-EOT
        The puppet-infrastructure face loads from an isolated enterprise
        environmentpath under /opt/puppetlabs/server. But it must recover hiera
        configuration from PE's environmentpath which defaults to
        /etc/puppetlabs/code/environments. If this value has been changed in
        puppet.conf it can be set here to ensure proper configuration recovery.
      EOT
    end

    option('--pe-environment ENVIRONMENT') do
      summary _('The environment to recover configuration from.')
      description <<-EOT
        The puppet-infrastructure face load from an isolated enterprise
        environment under /opt/puppetlabs/server. But it recovers hiera configuration
        from PE's environment as specified in the PE Node Groups. Commonly this
        is the 'production' environment, but if PE infrastructure has been configured
        in a different environment it may be specified here.
      EOT
    end

    when_invoked do |options|
      begin
        lock do
          conf = Puppet::Util::Pe_conf.new()
          conf.recover(options)

          pp_pe_conf = JSON.pretty_generate(conf.pe_conf)
          Puppet.debug(_("Generated the following user_data.conf: %{conf}") % { conf: pp_pe_conf })

          if conf.nodes_conf.size > 0
            pp_nodes_conf = JSON.pretty_generate(conf.nodes_conf)
            Puppet.debug(_("Generated the following node specific overrides: %{conf}") % { conf: pp_nodes_conf })
          end


          dir = options[:target_dir]
          conf.save(dir)

          conf
        end
      rescue Puppet::LockError
        Puppet.notice _("Puppet run already in progress") + "; " + _("aborting recover_configuration (%{lockfile_path} exists)") % { lockfile_path: lockfile_path }
        exit(17)
      end
    end

    when_rendering(:console) do |conf, options|
      target_dir = options[:target_dir] || Puppet::Util::Pe_conf::CONF_DIR
      node_output = nil
      if conf.nodes_conf.size > 0
        node_output = _("Saved node specific overrides to %{dir}/nodes")
      end
      pe_conf_output = _("Saved user_data.conf to %{dir}")
      [node_output, pe_conf_output].reject(&:nil?).join("\n") % { dir: target_dir }
    end
  end
end
