# coding: utf-8
require 'puppet/indirector/face'
require 'puppet_x/util/stringformatter'
require 'puppet_x/puppetlabs/meep/infra/plan_executor'
require 'puppet_x/puppetlabs/meep/infra/feature_flags'

Puppet::Face.define(:infrastructure, '1.0.0') do
  extend PuppetX::Puppetlabs::Meep::Infra::FeatureFlags

  def command_mappings
    hash = {
      'regenerate_agent_certificate'      => 'enterprise_tasks::agent_cert_regen',
      'regenerate_compiler_certificate'   => 'enterprise_tasks::secondary_cert_regen',
      'regenerate_replica_certificate'    => 'enterprise_tasks::secondary_cert_regen',
      'enable_ha_failover'                => 'enterprise_tasks::enable_ha_failover',
      'regenerate_primary_certificate'    => 'enterprise_tasks::primary_cert_regen',
      'regenerate_standalone_db_certificate' => 'enterprise_tasks::standalone_pg_cert_regen',
      'rebuild_certificate_authority'     => 'enterprise_tasks::rebuild_ca',
      'remove_old_postgresql_versions'    => 'enterprise_tasks::remove_old_postgresql_versions',
      'remove_old_pe_packages'            => 'enterprise_tasks::remove_old_pe_packages',
      'convert_legacy_compiler'           => 'enterprise_tasks::convert_legacy_compiler',
      'toggle_lockless_deploys'           => 'enterprise_tasks::toggle_lockless_deploys',
    }
    hash
  end

  def command_engine
    hash = {
      'enterprise_tasks::agent_cert_regen'               => 'orchestrator',
      'enterprise_tasks::secondary_cert_regen'           => 'orchestrator',
      'enterprise_tasks::enable_ha_failover'             => 'bolt',
      'enterprise_tasks::primary_cert_regen'             => 'bolt',
      'enterprise_tasks::rebuild_ca'                     => 'bolt',
      'enterprise_tasks::remove_old_postgresql_versions' => 'bolt',
      'enterprise_tasks::remove_old_pe_packages'         => 'orchestrator',
      'enterprise_tasks::convert_legacy_compiler'        => 'orchestrator',
      'enterprise_tasks::toggle_lockless_deploys'        => 'orchestrator',
    }
    hash
  end

  # @return [Hash] of replacement strings used to sanitize Bolt output
  # for the purpose of being self-referential for the listed puppet-infra
  # commands and actions.
  def _replacements
    replace_bolt = {'bolt plan run' => 'puppet infrastructure run'}
    command_mappings.merge(replace_bolt)
  end

  def _infra_run_internal_bolt(args)
    if args.size > 0
      bolt_action = args.join(' ')
    else
      # Homage to the original: https://github.com/puppetlabs/puppet/blob/92d93fc6ec3b47baae9fe2547b87e497384f6d19/lib/puppet/face/help.rb#L34-L48
      c = "@%#+*\n=-: .".split('')
      i = <<-'EOT'.gsub(/\s*/, '').to_i(36)
        15e99jic477co7gjmi3f268oqbt1lka9tsc6htvhsg33jxhepdssafqmyncai3pfw9vkw4z5znkb56bidcm5szbd2gpeexd7vjwdyfeyzgfjl9ig2bpnhl8or6bijn1yauzd
        jpd5z6fz4jvk3pb08n08ayywvq6e2yu1gj0onivtwg4swdsewn1eeoyzpwykzlb0gx75z1p4zzny88jv9xuel491rfzq2kupljb11ptvsevks1dwwhv6x4whdzl6pfivwxyj
        eg1c91g17hvmp77cc5qh55mxlvyyldj5tkldbsjg42fy204z2kg9yxkvvr4zifd4ndp044lqwdubw6bwvkwnwjrnb8ipd3bj3v40lewm1isczghbydh96hp70bnhsxhhoqlu
        uqxctpfqcvtb86cb8uen0xwf9blajvl64v7p74jvf93b9adlspdwijsi0fu0mpxcx8yci5qxhyuoww4nf7cvajm1ar7iwnh45cbnw5735qwfk3y9b1bg01lqkj01qidwprky
        dyp321xschakkon6q9i29vghjog2sop33yispg08x64ex1s153t0kiu0srspci4bl2c9ntnui8ublwrav4xix8izi71dgemeozdw01rmfsv7onhebip1w8f2forn6qzym9ae
        938c6gg9o4wht3nadiupbjll95z337pu4nztls0418euul6tg814hns4jrowm5aelrdyw0f1ajulbgiu6o69vp9ldqncti64ph8yeu2ruki4ki3a8ob1u6ud70lld2uhs74p
        ty5dbp4dcxygvp5ezlwfa0pei7h83srwzfifj168e2neueg2l0zq32gnvowzvznay30zpvysi3n6s22setp9cs00hajya3cpz2g670cj1mkdtz7xoqkbbaewc6ob6h4w3uz5
        1hfuy0ei86g8w2lv4i50vr57cnb383llam53tiq7rh6nl881qqw1m9di1bwp1jmqyrapfphbrl4xqxvp63udtor8nekeissb5tbfl2ee35ximujzliomkjkai3pw69lnvg30
        l3p1my1nkqg25fsp6mup3x9meh6qfcx1kyf1lrj4svow6vpew02ripj35y6dtg81mfukyozelptzn38sbo8ytt1bfhr8nh2eiotuqyv0wrfpxzbin1o4ix05xt39aqm9rzdt
        9w7q6dwk741z17hqg87t22f75532d0xgutalfi7j8wv81dsl2c2h9hu
      EOT
      697.times{i,x=i.divmod(594);a,b=x.divmod(54);print(c[a]*b)}
      puts 'USAGE: puppet infra run boltymcboltface -- <any bolt command or flag to be passed directly to something that is definitely not bolt>'
      puts
      puts '⚠️  DO NOT USE THIS WITH A PRODUCTION/CUSTOMER SYSTEM  ⚠️'
      puts '⚠️  IF YOU NEED BOLT, INSTALL IT NORMALLY: https://puppet.com/docs/bolt/latest/bolt_installing.html  ⚠️'
      exit 1
    end

    PuppetX::Util::Bolt.run_bolt(bolt_action.strip, _replacements)
  end

  def _infra_show_plan(command, options)
    plan = command_mappings[command]
    bolt_action = "plan show #{plan}"
    PuppetX::Util::Bolt.run_bolt(bolt_action.strip, _replacements)
  end

  def _infra_run_plan(command, args, options)
    plan = command_mappings[command]

    options[:debug] = true if Puppet[:debug]
    options[:trace] = true if Puppet[:trace]

    args |= ['force=true'] if options[:force]
    raise ArgumentError, _("Multiple values for 'force' argument given") if args.count{ |s| s =~ /^force=/ } > 1
    engine = command_engine[plan]
    engine = 'bolt' if options[:use_ssh] || options[:use_winrm]
    options[:transport] = 'winrm' if options[:use_winrm]

    log_context =  capture_cli_logs('puppet_infra_run', capture_cli: false)

    PuppetX::Puppetlabs::Meep::Infra::PlanExecutor.run(
      plan,
      params: args,
      options: options,
      replacements: _replacements,
      engine: engine,
      log_context: log_context
    )
  end

  ### Puppet face action
  action :run do
    summary _('Run PE maintenance commands.')
    description(
      PuppetX::Util::String::Formatter.wrap_no_indent(_('Run PE maintenance commands'))
    )
    arguments "\nValid commands:\n" + @face.command_mappings.keys.join("\n")

    # Provide a sub-set of bolt options to be passed in
    option('-h', '--help') do
      summary _('Displays this help text.')
    end
    option('--[no-]host-key-check') do
      summary _('Enables or disables automatic host key checking.')
    end
    option('--[no-]ssl-verify') do
      summary _('Enables or disables remote host SSL verification.')
    end
    option('--connect-timeout TIMEOUT') do
      summary _('Sets a connection timeout value.')
    end
    option('--inventoryfile FILE') do
      summary _('Sets a custom path to your inventory file.')
    end
    option('--sudo-password PASSWORD') do
      summary _('Pass your superuser password through to your run command.')
    end
    option('--run-as USER') do
      summary _('Run your command as the specified user.')
    end
    option('--private-key KEY') do
      summary _('Pass your private key through to your run command.')
    end
    # -u/--user doesn't need to be added because it is a Puppet default
    # --user  option will set Puppet[:user]. Adding --username to use for bolt target connection
    option('--username USER') do
      summary _('Pass your username through to your run command to connect to target over ssh/winrm')
    end
    option('--password PASSWORD') do
      summary _('Pass your password through to your run command.')
    end
    option('--tmpdir DIRECTORY') do
      summary _('Directory that Bolt will use to upload and execute temporary files on the target. Defaults to $TMPDIR if set or /tmp if not.')
    end
    option('--[no-]tty') do
      summary _('Request a pseudo TTY on nodes that support it.')
    end
    option('--use-ssh') do
      summary _('Use ssh for transport rather than relying on the Orchestrator.')
    end
    option('--use-winrm') do
      summary _('Use winrm for transport rather than relying on the Orchestrator.')
    end
    option('--pe-environment ENVIRONMENT') do
      summary _('The environment to be used when running tasks and plans.')
    end
    option('--token-file PATH') do
      summary _('Path to a token file')
      description <<-EOT
        Path to an RBAC token to use for authentication when running plans with the Orchestrator.
      EOT
    end
    option("--force") do
      summary _("Skip all node type verification.")
      description <<-EOT
        Skip all node type verification when running a plan.
      EOT
    end

    when_invoked do |command, *args|
      options = args.pop

      if command == 'boltymcboltface'
        status, _output = _infra_run_internal_bolt(args)
      elsif command_mappings.keys.include?(command) && options[:help]
        status, _output = _infra_show_plan(command, options)
      elsif command_mappings.keys.include?(command)
        status, _output = _infra_run_plan(command, args, options)
      elsif command.nil? || command.empty? || command == "help" || (command.is_a?(Hash) && command[:help])
        puts Puppet::Face[:help, :current].help('infrastructure',:run)
        return
      else
        puts _('Expected subcommand to be one of the available plans:'), command_mappings.keys
        return
      end
      exit status.exitstatus
    end
  end
end
