class Puppet::Util::Pe_conf
  class Recover
    module Puppetdb

      # Retrieves all nodes that have a "core infra" profile applied to it.
      def self.get_pe_infra_nodes
        ast = <<-EOS
          resources[certname] {
            type = 'Class' and
            (
              title = 'Puppet_enterprise::Profile::Certificate_authority' or
              title = 'Puppet_enterprise::Profile::Master' or
              title = 'Puppet_enterprise::Profile::Console' or
              title = 'Puppet_enterprise::Profile::Puppetdb'
            ) and
            nodes { deactivated is null and expired is null }
          }
        EOS
        self.query(ast)
      end

      # Retrieves all nodes that have a given "Puppet_enterprise::Profile" profile applied to it.
      def self.get_pe_infra_nodes_with_class(classname)
        ast = ['from', 'resources',
                ['extract', ['certname'],
                  ['and',
                    ['=', 'type', 'Class'],
                    ['=', ['node', 'active'], true],
                    ['=', 'title', "Puppet_enterprise::Profile::#{classname}"]
                  ]
                ]
              ]

        self.query(ast)
      end

      # Retrieves the last known facts for a given certname
      #
      # Returns a hash like so:
      #
      #    {
      #       'name': 'platform_tag',
      #       'value': 'el-7-x86_64',
      #    },
      #    {
      #      'name': 'os',
      #      'value': {
      #        'architecture': 'x86_64',
      #         ...
      #       }
      #    }
      #
      # @param certname    [String] The certificate name of the node
      #
      # @return [Array] A list of hashes mapping with keys: name, value
      def self.get_facts_for_node(certname)
        ast = <<-EOS
          facts[name, value] {
            certname = '#{certname}'
          }
        EOS
        self.query(ast)
      end

      # Retrieves the base `puppet_enterprise` class and their parameters
      # as applied to any of the "core" infra nodes (master, console, puppetdb)
      #
      # @return [Array] An array containing hashes of the puppet_enterprise class
      #                 and their parameter values.
      def self.get_base_pe_class
        ast = <<-EOS
          resources[certname, parameters] {
            type = 'Class' and
            title = 'Puppet_enterprise' and
            certname in resources[certname] {
              type = 'Class' and
              (
                title = 'Puppet_enterprise::Profile::Master' or
                title = 'Puppet_enterprise::Profile::Console' or
                title = 'Puppet_enterprise::Profile::Puppetdb'
              ) and
              nodes { deactivated is null and expired is null }
            } and
            nodes { deactivated is null and expired is null }
          }
        EOS
        self.query(ast)
      end

      # Retrieves all resources and their parameters that are applied to
      # a given node.
      #
      # Example data returned:
      #  [
      #    {
      #      "title": "Puppet_enterprise::Profile::Master",
      #      "parameters": {
      #        "allow_unauthenticated_ca": true, 
      #        ...
      #      }
      #    },
      #    ...
      #  ]
      #
      #
      #
      # @param certname    [String] The certificate name of the node
      # @param namespace   [String] The resource namespace to match against
      #
      # @return [Array] A list of hashes per title containing the parameters and values
      def self.get_resources_for_node(certname, namespace)
        ast = <<-EOS
          resources[title, parameters] {
            type = 'Class' and
            certname = '#{certname}' and
            title ~ '#{namespace}' and
            nodes { deactivated is null and expired is null }
          }
        EOS
        self.query(ast)
      end

      private

      # Private wrapper around Puppetdb.query_puppetdb due to
      # the code being installed by the puppetdb package.
      #
      # If this was a require at the top of the file, then it would be
      # auto loaded / required as part of the install process, before
      # the puppetdb package was installed, producing an error.
      def self.query(pql)
        require 'puppet/util/puppetdb'
        Puppet::Util::Puppetdb.query_puppetdb(pql)
      end
    end
  end
end
