module PuppetX
  module Util
    class ClassificationError < StandardError
    end

    class NodeGroupNotFoundError < ClassificationError
      attr_reader :group_name

      def initialize(group_name)
        super(_("The %{group_name} node group could not be found.") % { group_name: group_name } + " " +  _("Verify that it exists in the node classifier."))
        @group_name = group_name
      end
    end

    class MissingClassError < ClassificationError
      attr_reader :group_name, :class_name

      def initialize(group_name, class_name)
        super(_("The %{group_name} node group is not configured with the %{class_name} class.") % { group_name: group_name, class_name: class_name } + " " + _("Verify that it is configured correctly in the node classifier."))
        @group_name = group_name
        @class_name = class_name
      end
    end

    class CreateNodeGroupError < ClassificationError
      attr_reader :group_name, :parent_id

      def initialize(group_name, parent_id)
        super(_("Could not create the %{group_name} node group.") % { group_name: group_name } + " " + _("Verify that the Console Services are running and have no relevant alerts using the 'puppet infra status' command."))
        @group_name = group_name
        @parent_id = parent_id
      end
    end

    class GetNodeGroupsError < ClassificationError
      def initialize()
        super(_("Could not load node groups from the node classifier.") + " " + _("Verify that the Console Services are running and have no relevant alerts using the 'puppet infra status' command."))
      end
    end

    class PinNodeError < ClassificationError
      attr_reader :group_name, :group_id, :node_certname

      def initialize(group_name, group_id, node_certname)
        super(_("Unable to pin %{node_certname} to the %{group_name} node group (id=%{group_id}).") % { node_certname: node_certname, group_name: group_name, group_id: group_id } + " " + _("Verify that the Console Services are running and have no relevant alerts using the 'puppet infra status' command.") )
        @group_name = group_name
        @group_id = group_id
        @node_certname = node_certname
      end
    end

    class UnpinNodeError < ClassificationError
      attr_reader :group_name, :group_id, :node_certname

      def initialize(group_name, group_id, node_certname)
        super(_("Unable to unpin %{node_certname} from the %{group_name} node group.") % { node_certname: node_certname, group_name: group_name } + " " + _("Verify that the Console Services are running and have no relevant alerts using the 'puppet infra status' command."))
        @group_name = group_name
        @group_id = group_id
        @node_certname = node_certname
      end
    end

    class UpdateClassificationParametersError < ClassificationError
      attr_reader :group_name, :group_id, :class_name, :parameter_names

      def initialize(group_name, group_id, class_name, parameter_names)
        super(_("Unable to set parameters (%{parameters}) for the class %{class_name} in the node group %{group_name}. (%{group_id}).") % { parameters: parameter_names.join(','), class_name: class_name, group_name: group_name, group_id: group_id } + " " + _("Verify that the Console Services are running and have no relevant alerts using the 'puppet infra status' command, and that the node group is present in the node classifier."))

        @group_name = group_name
        @group_id = group_id
        @class_name = class_name
        @parameter_names = parameter_names
      end
    end

    class GetClassificationError < ClassificationError
      attr_reader :node_certname

      def initialize(node_certname)
        super(_("Unable to classify %{node_certname}.") % { node_certname: node_certname } + " " + _("Verify that there are no conflicts in the node classifier configuration."))
        @node_certname = node_certname
      end
    end

  end
end
