require 'puppet_x/puppetlabs/meep/infra/lookup'
require 'puppet_x/puppetlabs/meep/configure/validator'
require 'puppet_x/util/bolt'

module PuppetX
  module Util
    module LEI
      extend PuppetX::Puppetlabs::Meep::Infra::Lookup

      def get_nodes_with_role(role)
        # If puppetdb would be required at the top of the file,
        # then it would be autoloaded/required as part of the install process,
        # before the puppetdb package was installed, producing an error. Also,
        # this allows specs to work.
        require 'puppet/util/puppetdb'
        if role == 'compile_master'
          # the resource entity will return multiple entries for a certname, but type/title are
          # unique per-certname so this will only return one copy of any certname
          query = "resources[certname] { type = 'Class' and title = 'Puppet_enterprise::Profile::Master' }"
          result = Puppet::Util::Puppetdb.query_puppetdb(query)
          result = result.select { |n| is_legacy_compiler?(n['certname']) }
        elsif role == 'pe_compiler'
          # Use facts.trusted... over the top-level trusted... to hit the right PuppetDB index
          query = "inventory[certname] { facts.trusted.extensions.pp_auth_role = 'pe_compiler' }"
          result = Puppet::Util::Puppetdb.query_puppetdb(query)
        else
          query = "resources[certname] { type = 'Class' and title = 'Puppet_enterprise::Profile::#{role.capitalize}' }"
          result = Puppet::Util::Puppetdb.query_puppetdb(query)
        end
        result.map { |val| val['certname'] }
      end

      def is_legacy_compiler?(host)
        masters = get_nodes_with_role('master')
        # Have to use the face here because it requires enterprise_hiera_yaml that is defined in configure
        replicas = get_nodes_with_role('primary_master_replica')
        primary = Puppet::Face[:infrastructure, :current].puppet_lookup('puppet_enterprise::puppet_master_host')

        masters.include?(host) && !replicas.include?(host) && host != primary && !is_pe_compiler?(host)
      end

      def is_pe_compiler?(host)
        pe_compilers = get_nodes_with_role('pe_compiler')
        pe_compilers.include?(host)
      end

      def get_compiler_type(host)
        is_pe_compiler?(host) ? 'pe_compiler' : (is_legacy_compiler?(host) ? 'compiler' : 'invalid')
      end

      def validate_compiler(host, options)
        # Skip node validation when --force is used
        return true if options[:force]
        state = {}
        rules = []

        state[:compiler_host] = host
        state[:compiler_type] = get_compiler_type(host)

        rules << PuppetX::Puppetlabs::Meep::Configure::CompilerUpgrade
        validator = PuppetX::Puppetlabs::Meep::Configure::Validator.new(state)
        validator.register(rules)

        validator.log_and_fail! unless validator.valid?
        validator.valid?
      end
    end
  end
end
