require 'time'
require 'puppet_x/util/orchestrator/connection'

module PuppetX
  module Util
    class Orchestrator

      # Abstraction for working with a single Orchestrator Job.
      # This handles calls to:
      #
      # /orchestrator/v1/jobs/#
      # /orchestrator/v1/jobs/#/nodes
      # /orchestrator/v1/jobs/#/events
      #
      # And keeps the state of a given job, and its nodes/events as requested.
      #
      # See:
      #
      # https://puppet.com/docs/pe/2019.3/orchestrator_api_jobs_endpoint.html
      # https://puppet.com/docs/pe/2019.3/orchestrator_api_events.endpoint.html
      #
      # for Orchestrator API details.
      #
      # This class is used by all the Runner implementations.
      #
      # It's variation PlanJob (see below) is used by PlanRunner to interact
      # with the separate endpoint for a plan job.
      class Job < Connection
        # The Job id
        attr_accessor :id

        # The seconds to wait between failed attempts to query the orchestrator.
        attr_accessor :poll_interval

        # The total time to wait for successfully retrieving a Job from the
        # Orchestrator before raising an error.
        attr_accessor :timeout

        # Cached or preloaded response bodies for the various calls.
        attr_accessor :job_body, :events_body, :nodes_body


        def initialize(service_config:, rbac_token:, job_id:, poll_interval: 1, timeout: 300)
          super(service_config, rbac_token)
          self.id = job_id.to_s
          self.poll_interval = poll_interval
          self.timeout = timeout
          self.job_body = {}
          self.events_body = {}
          self.nodes_body = {}
        end

        # Because we are often interacting with the Orchestrator to run Puppet
        # and reconfigure PE, there are cases where we cannot assume that the
        # Orchestrator and RBAC are both up. The {Job#wait_for_job_status()} requests
        # every +poll_interval+ until +timeout+ or a successful Job body is
        # fetched and set as {Job#job_body}.
        #
        # This does not determine that the job has finished *running*; it just
        # ensures that we have successfully retrieved the current job state.
        #
        # To use this method repeatedly, in order to poll status while 'running'
        # until it reaches 'finished' or some other stop point, set +reload+
        # to true so that +job_body+ is cleared each time...
        #
        # @param reload [Boolean] set to true to clear job_body first.
        # @param poll_interval [Numeric] seconds to wait before retrying when a
        # non ok response code is received.
        # @param timeout [Numeric] total seconds to attempt querying the Orchestrator
        # for a successful response before giving up.
        # @return [Boolean] true when loaded.
        # @raise [PuppetX::Util::OrchestratorJobTimeoutError] if unable to load
        # before +timeout+.
        # @raise [PuppetX::Util::OrchestratorEmptyResponseError] if somehow we
        # get a succesful response with a nil body...previous code was testing
        # this case...
        def wait_for_job_status(reload: false, poll_interval: self.poll_interval, timeout: self.timeout)
          start_time = Time.now
          self.job_body = {} if reload
          while loading?
            if (Time.now - start_time) > timeout
              # Unable to get a successful request from the Orchestrator for
              # #{Job#timeout} seconds.
              raise(PuppetX::Util::OrchestratorJobTimeoutError.new(timeout, id))
            end

            response = _get_job

            if response.code.to_i != 200
              sleep(poll_interval)
            else
              raise PuppetX::Util::OrchestratorEmptyResponseError.new() if response.body.nil? || response.body == ''
              self.job_body = JSON.parse(response.body)
            end
          end
          !loading?
        end

        # Wait until the job is no longer running.
        def wait_for_job_completion(poll_interval: self.poll_interval, timeout: self.timeout)
          while in_progress?
            wait_for_job_status(reload: true, poll_interval: poll_interval, timeout: timeout)

            if in_progress?
              sleep(poll_interval)
            end
          end
          self
        end

        # @return [Boolean] true if not in a final state.
        def in_progress?
          !(failed? || stopped? || finished? || successful?)
        end

        # @return [Boolean] true when we've successfully loaded (or been given)
        # a {Job#job_body}.
        def loading?
          job_body.empty?
        end

        # @return [Boolean] true when the fetched job state indicates the job
        # is still running.
        def running?
          state == 'running'
        end

        # @return [Boolean] true when the fetched job state indicates the job
        # was stopped (by an /orchestrator/v1/command/stop for example...).
        def stopped?
          state == 'stopped'
        end

        # @return [Boolean] true if execution of the job failed.
        def failed?
          state == 'failed'
        end

        # @return [Boolean] true when the job finished executing (either Puppet or
        # the Task completed without raising an error).
        def finished?
          state == 'finished'
        end

        # For a Puppet job or Task, if the state is finished, it was successful.
        def successful?
          finished?
        end

        def state
          job_body['state']
        end

        def _get_task_started(status)
          (status || []).map { |s| s['enter_time'] }.sort.first
        end

        # @return [String,nil] time the job started running.
        def started
          _get_task_started(job_body['status'])
        end

        # @return [String] time value of the timestamp field.
        def timestamp
          job_body['timestamp']
        end

        # Helper to construct the url to the jobs endpoint.
        def url
          "#{service_config[:url]}/v1/jobs/#{id}"
        end

        # Because we are often interacting with the Orchestrator to run Puppet
        # and reconfigure PE, there are cases where we cannot assume that
        # Orchestrator and RBAC are both up, so we don't check for a 200
        # response in this method. Instead we return the response itself, and leave
        # that test to {Job#wait_for_job_status()}
        #
        # @return [Net::HTTPResponse]
        def _get_job
          uri = URI(Puppet::Util.uri_encode(url))
          response = orch_connection.get(uri, headers: orch_request_headers)
          response
        end

        # @param endpoint [Symbol] :events, :nodes...
        # @return [Hash] parsed response body from the requested endpoint.
        def _load(endpoint)
          uri = URI(Puppet::Util.uri_encode("#{url}/#{endpoint.to_s}"))
          response = orch_connection.get(uri, headers: orch_request_headers)
          check_orch_response(response, 200)
          JSON.parse(response.body)
        end

        def _load_events
          self.events_body = _load(:events)
        end

        # The list of events currently associated with the job.
        #
        # Since a job is started asynchronously with a /v1/command/deploy (or task,
        # or plan_job) call, and may consist of several actions across multiple nodes,
        # these events queue under the /v1/jobs/#/events or /v1/plan_jobs/#/events
        # endpoint.
        #
        # @return [Array<Hash>] the list of items from the events_body, sorted by timestamp.
        def events(refresh: false)
          _load_events if events_body.empty? || refresh

          events_body['items'].sort { |a,b| Time.parse(a['timestamp']) <=> Time.parse(b['timestamp']) }
        end

        # @return [Array<Hash>] off all events that are newer than the passed +last_event+.
        def events_since(last_event)
          if last_event.nil?
            events(refresh: true)
          else
            next_index = events(refresh: true).find_index(last_event) + 1
            events[next_index..-1]
          end
        end

        def _load_nodes
          self.nodes_body = _load(:nodes)
        end

        # @return [Hash] of original calling options for the job.
        def options
          job_body['options'] || {}
        end

        # @return [Array] of nodes the job is acting on.
        def scope
          if options['scope']
            options['scope']['nodes'] || []
          elsif !nodes_body.empty?
            nodes_body['items'].map { |node_item| node_item['name'] }
          else
            []
          end
        end

        # @return [String] the Job's "type" field.
        def type
          job_body['type']
        end

        # The thing the job is doing.
        # For a job that is running the task, it's the task name.
        # For a job running a command, it's the command.
        # For a wait job, we return the string 'until available'
        #
        # @return [String]
        def action
          case type
          when 'plan_task', 'task'
            options['task']
          when 'plan_command'
            options['command']
          when 'plan_wait'
            'until available'
          else
            Puppet.debug(_("Unknown action for type %{type} for job %{job}") % { type: type, job: job_body })
          end
        end

        # From the current job status body, count the nodes in the node_states hash
        # they are neither running nor finished.
        #
        # @return [Integer] count of node_states nodes that aren't 'finished' or 'running'.
        def node_failure_count
          node_states = job_body['node_states'] || {}
          node_states.select { |k,v| !['running', 'finished'].include?(k) }.values.sum
        end

        # @return [Float] secs task took to complete.
        def duration
          loading? || running? ?
            0 :
            node_items.map { |n| n['duration'] }.max
        end

        # This is the 'items' array from an /orchestrator/v1/jobs/#/nodes call.
        #
        # The Job that was executed might have been either a puppet run or a
        # task. See the Orchestrator API docs for details, but each hash in the
        # array represents results of execution on one target, and should
        # include the following fields (among others):
        #
        #   'name': certname it executed on
        #   'state': a string representing the end state of the execution
        #   'result':
        #      a well behaved Task will return a Hash, but could be anything.
        #      a Puppet run will return a Hash of details about the run.
        #
        # @return /orchestrator/v1/jobs/+id+/nodes items array.
        def node_items
          _load_nodes if nodes_body.empty?

          nodes_body['items'] || []
        end

        # @return [Array<Hash>] just those entries from the nodes endpoint
        # items array that are in a failed or errored state.
        def failed_node_items
          node_items.select { |i| ['failed', 'errored'].include?(i['state']) }
        end

        # @return [Array<Hash>] The details hashes from failed nodes only.
        # These should have the report-urls for failed puppet nodes, for example.
        def failed_node_details
          failed_node_items.map do |i|
            details = i['details'] || {}
            details.merge('name' => i['name'])
          end
        end

        # Tasks will generally return an _error entry in their result.
        # This method will return a Hash keyed by node cert, with the
        # corresponding error hash for a value:
        #
        #   {
        #     'node.cert' => {
        #       'msg'     => '_error.msg',
        #       'kind'    => '_error.kind',
        #       'details' => '_error.details',
        #     }
        #   }
        #
        def task_errors
          failed_node_items.each_with_object({}) do |f, hash|
            result = f['result'] || {}
            error = result['_error'] || {}
            hash[f['name']] = {
              'msg'     => error['msg'],
              'kind'    => error['kind'],
              'details' => error['details'],
            }
          end
        end

        def ==(other)
          other.is_a?(PuppetX::Util::Orchestrator::Job) &&
            other.id == id
        end

        def hash
          id.hash
        end
      end

      # Plans has a similar but slightly different API for a started Job.
      # This class implements the differences over Job.
      #
      # NOTE: there is no +nodes+ endpoint for plan_jobs...
      #
      # See:
      #
      # https://puppet.com/docs/pe/2019.3/orchestrator_api_plan_jobs_endpoint.html
      # https://puppet.com/docs/pe/2019.3/orchestrator_api_events.endpoint.html
      #
      # for Orchestrator API details.
      class PlanJob < Job
        def url
          "#{service_config[:url]}/v1/plan_jobs/#{id}"
        end

        def _load_nodes
          raise(RuntimeError, _("There is no 'nodes' endpoint for plan_jobs."))
        end

        # @return [Float] secs between job start and last state.
        def duration
          start_time_string = job_body['created_timestamp']
          start_time = start_time_string.nil? ? Time.now : Time.parse(start_time_string)
          end_time_string = job_body['finished_timestamp']
          end_time = end_time_string.nil? ? start_time : Time.parse(end_time_string)
          end_time - start_time
        end

        def failed?
          state == 'failure'
        end

        def successful?
          state == 'success'
        end

        # If the plan failed, the job's result Hash will have msg, kind and details.
        def error
          job_body['result'] || {}
        end

        # @return [String,nil] time the job started running based on the time the
        # earliest task started running.
        def started
          status = job_body['status'] || {}
          status.map { |_, task_status| _get_task_started(task_status) }.sort.first
        end
      end
    end
  end
end
