require 'puppet'
require 'puppet_x/util/orchestrator_error'
require 'puppet_x/util/orchestrator/connection'
require 'puppet_x/util/orchestrator/job'

module PuppetX
  module Util
    class Orchestrator

      # Abstract runner for Orchestrator command API.
      #
      # Basic use of a runner:
      #
      #   runner = Runner.new(orchestrator_service_config_hash, rbac_token)
      #   runner.run(args)
      #
      # The run() method is the principal interface. It handles
      # starting the appropriate command endpoint (via start_command()),
      # and then works with the returned Job instance asynchronously until
      # it is complete. If the job was successfull it calls the
      # _get_job_result() function to return whatever the Runner considers
      # appropriate for the job type. The _handle_job_failure() function is
      # called if the job failed, and _handle_error() if communication with the
      # Orchestrator raised an error.
      #
      # Concrete subclasses must implement
      #
      #  * start_command()
      #  * _handle_job_failure()
      #
      # The two additional methods intended for use by other classes are
      # helpers for constructing a Job or PlanJob instance wired with the
      # Runner's Connection info:
      #
      #  * job_for(id)
      #  * plan_job_for(id)
      #
      class Runner < Connection

        # Used as a safe timeout for task and plan execution to bypass
        # the defaults of forty minutes and one hour respectively. (Or
        # any other default that may have been manually set by the user for the
        # Orchestrator service).
        ONE_YEAR_IN_SECONDS = (365*24*60*60)

        # String, 'puppet', 'task', 'plan', representing the type of
        # Orchestrator command being executed.
        attr_reader :command_type

        # PuppetX::Puppetlabs::Meep::Util::LogContext instance holding
        # common details about log file paths for puppet/task runners.
        attr_reader :log_context

        # Helper method to produce a Job instance for the given job_id
        # without worrying about service_config/rbac_token.
        def job_for(job_id)
          PuppetX::Util::Orchestrator::Job.new(
            job_id: job_id,
            service_config: service_config,
            rbac_token: rbac_token,
            poll_interval: orch_status_wait_seconds,
          )
        end

        # Helper method to produce a PlanJob instance for the given job_id
        # without worrying about service_config/rbac_token.
        def plan_job_for(job_id)
          PuppetX::Util::Orchestrator::PlanJob.new(
            job_id: job_id,
            service_config: service_config,
            rbac_token: rbac_token,
            poll_interval: orch_status_wait_seconds,
          )
        end

        # Wait until a started job has reached a non-running state (finished,
        # failed, stopped...), or we hit some exception like a timeout polling
        # for job state.
        #
        # @return [String] Final job state.
        def _wait_for_orch_job(job:, display_scope:, task: nil, **_kwargs)
          Puppet.notice(
            _("Running %{task}%{job_type} on %{display_scope}.") % { display_scope: display_scope, job_type: command_type, task: (task.nil? ? '' : "#{task} ") }
          )
          Puppet.notice(
            _("To monitor detailed progress run:\n") +
            "    puppet job show %{job_id}" % { job_id: job.id }
          )

          job.wait_for_job_completion(poll_interval: orch_status_wait_seconds)
        end

        def _handle_job_result(job)
          message = _("Overall job status: %{last_state}...") % { last_state: job.state }
          if job.successful?
            Puppet.info(message)
          else
            Puppet.err(message)
          end
          
          STDOUT.flush

          if job.stopped?
            raise PuppetX::Util::OrchestratorJobStoppedError.new(job.id)
          elsif job.failed?
            _handle_job_failure(job)
          else
            job.state
          end
        end

        def _wait_for_job_to_finish(job, args)
          orchargs = { job: job }.merge(args)
          _wait_for_orch_job(**orchargs)
          _handle_job_result(job)
        end

        def _handle_error(error, args)
          Puppet.err(_("Error during orchestrated %{command_type} run on %{display_scope}." % { command_type: command_type, display_scope: args[:display_scope] }))
          raise error
        end

        # If our scope produces an empty list of targets (a Puppetdb query that finds no nodes,
        # for example), we may or may not want to raise the Orchestrator empty-target error
        # depending on whether the caller expected this.
        def _handle_empty_target(error, args)
          if !args[:allow_empty]
            Puppet.err(error)
            raise error
          end

          message = [
           _("Skipping %{command_type} run on %{display_scope}.") % { command_type: command_type, display_scope: args[:display_scope] },
           _("(nothing to do)"),
          ]
          Puppet.notice(message.join(' '))
        end

        def _save_log(job, args)
          # Both PuppetRunner and TaskRunner implement this method.
          # The PlanRunner handles logging separately.
        end

        def _get_job_result(job)
          job.state
        end

        def run(args)
          begin
            save_log = true
            job = start_command(**args)
            _wait_for_job_to_finish(job, **args)
            _get_job_result(job)
          rescue PuppetX::Util::OrchestratorEmptyTargetError => e
            _handle_empty_target(e, **args)
            save_log = false
            return :skipped # No nodes found to run on
          rescue PuppetX::Util::OrchestratorError => e
            _handle_error(e, **args)
          ensure
            _save_log(job, **args) if save_log
          end
        end
      end
    end
  end
end
