# This class configures the pe-workflow-service.
#
# @param certname [String] The name of the workflow service SSL certificate.
# @param service_name [String] The name of the service.
# @param database_host [String] The hostname of the database for this service.
# @param database_port [Port] The port that the PostgreSQL database host for the
# workflow service is listening on.
# @param database_password [Optional[String]] The password to use when using password
# auth on the workflow service database.
# @param database_properties [String] Additional properties to add on to the
# "database.subname" value.
# @param disable_service [Boolean] Whether the service should be running or not.
# By default, the workflow_service is disabled. Setting this to true in the PE Master
# node group allows this service to start on the primary. It is always set to false
# on the replica.
# @param service_stop_retries [Integer] Number of seconds to wait during shutdown after
# the TERM signal is sent before sending a KILL signal to the process.
# @param service_start_timeout [Integer] Number of seconds to wait during startup for
# the service to finish booting before timing out.
# @param ssl_listen_address [String] The network interface address used for SSL
# connections to the service.
# @param ssl_listen_port [Integer] The port used for SSL connections to the service.
# @param puppetdb_host Array[String] Array of the hosts running puppetdb.
# @param puppetdb_port Array[Integer] Array of the ports that puppetdb is listening on
# matching the puppetdb_host array.
# @param java_args [Hash] Java args to use for the service. Defaults to setting the
# heap size.
# @param flight_recorder_enabled [Boolean] Enable/disable Java Flight Recorder for
# all Java services.
# @param flight_recorder_settings [String] Settings value for JFR. The 'default'
# settings is low overhead, 'profile' gives more data.
# @param flight_recorder_stackdepth [Integer] Number of stack frames to capture.
# Default is 128 to account for Clojure/JRuby's deeply nested function calls.
# @param flight_recorder_maxage [String] Maximum amount of time to keep recorded data
# for each service. Can be specified by s (seconds), m (minutes), h (hours), or
# d (days) suffixes.
# @param flight_recorder_maxsize [String] Maximum size of recorded data files for
# each service. Can be specified with k (kilobytes), m (megabytes) and g (gigabytes)
# suffixes.
# @param flight_recorder_repository [String] Directory to save JFR chunks while recording.
# @param flight_recorder_filename [String] File to save completed recording.
# @param disable_string_deduplication [Boolean] When using the default G1 garbage
# collector, this disables the use of string deduplication for optimization.
# @param enable_gc_logging [Boolean] Enable the garbage collector log.
class puppet_enterprise::profile::workflow_service (
  String $certname                        = $facts['clientcert'],
  String $service_name                    = 'pe-workflow-service',
  String $database_host                   = $puppet_enterprise::workflow_database_host,
  Variant[String, Integer] $database_port = $puppet_enterprise::database_port,
  Optional[String] $database_password     = $puppet_enterprise::workflow_database_password,
  String $database_properties             = $puppet_enterprise::database_properties,
  Boolean $disable_service                = true,
  Integer $service_stop_retries           = 60,
  Integer $service_start_timeout          = 300,
  String $ssl_listen_address              = $puppet_enterprise::params::ssl_address,
  Integer $ssl_listen_port                = $puppet_enterprise::workflow_service_port,
  Array[String] $puppetdb_host            = $puppet_enterprise::puppetdb_hosts_array,
  Array[Integer] $puppetdb_port           = $puppet_enterprise::puppetdb_ports_array,
  Hash $java_args                         = $puppet_enterprise::params::workflow_service_java_args,
  Boolean $flight_recorder_enabled        = $puppet_enterprise::flight_recorder_enabled,
  String $flight_recorder_settings        = $puppet_enterprise::flight_recorder_settings,
  Integer $flight_recorder_stackdepth     = $puppet_enterprise::flight_recorder_stackdepth,
  String $flight_recorder_maxage          = $puppet_enterprise::flight_recorder_maxage,
  String $flight_recorder_maxsize         = $puppet_enterprise::flight_recorder_maxsize,
  String $flight_recorder_repository      = '/var/log/puppetlabs/workflow-service/jfr',
  String $flight_recorder_filename        = '/var/log/puppetlabs/workflow-service/last_recording.jfr',
  Boolean $disable_string_deduplication   = $puppet_enterprise::disable_string_deduplication,
  Boolean $enable_gc_logging              = true,
) inherits puppet_enterprise {
  $container      = 'workflow-service'
  $package_name   = 'pe-workflow-service'
  $user           = 'pe-workflow-service'
  $group          = 'pe-workflow-service'
  $etc_dir        = "/etc/puppetlabs/${container}"
  $confdir        = "${etc_dir}/conf.d"
  $webserver_conf = "${confdir}/webserver.conf"
  $ssl_dir        = "${puppet_enterprise::server_data_dir}/workflow-service/ssl"
  $local_ssl_cert = "${ssl_dir}/${certname}.cert.pem"
  $local_ssl_key  = "${ssl_dir}/${certname}.private_key.pem"
  $puppetdb_url   = "https://${puppetdb_host[0]}:${puppetdb_port[0]}"
  
  if $puppet_enterprise::database_ssl and $puppet_enterprise::database_cert_auth {
    $client_pk8_key          = "${ssl_dir}/${certname}.private_key.pk8"
    $client_cert             = "${ssl_dir}/${certname}.cert.pem"
    $ssl_database_properties = "${database_properties}&sslkey=${client_pk8_key}&sslcert=${client_cert}"
  } else {
    $ssl_database_properties = $database_properties
  }

  $common_hocon_settings = {
    ensure  => present,
    notify  => Service[$service_name],
    require => Package[$package_name],
  }

  $common_file_parameters = {
    ensure  => file,
    owner   => $user,
    group   => $group,
    mode    => '0640',
    require => Package[$package_name],
  }

  ### Package ###
  include puppet_enterprise::packages
  Package <| tag == 'pe-workflow-packages' |> {
    before => [Class[puppet_enterprise::trapperkeeper::workflow]],
  }

  ### Make copy of cert/key in service ssl directory ###
  puppet_enterprise::certs { 'pe-workflow-service::ssl':
    certname      => $certname,
    container     => $container,
    cert_dir      => $ssl_dir,
    append_ca     => false,
    make_pk8_cert => true,
    notify        => Service[$service_name],
    require       => Package[$package_name],
  }

  ### global.conf ###
  puppet_enterprise::trapperkeeper::global_settings { $container:
    certname    => $certname,
    ssl_cert    => $local_ssl_cert,
    ssl_key     => $local_ssl_key,
    localcacert => $puppet_enterprise::params::localcacert,
    require     => Package[$package_name],
  }

  ### analytics client settings ###
  puppet_enterprise::trapperkeeper::analytics_client_settings { $container:
    container => $container,
    owner     => $user,
    notify    => Service[$service_name],
    require   => Package[$package_name],
  }

  ### workflow service configuration ###
  class { 'puppet_enterprise::trapperkeeper::workflow':
    service_name        => $service_name,
    database_host       => $database_host,
    database_port       => $database_port,
    database_password   => $database_password,
    database_properties => $ssl_database_properties,
    puppetdb_url        => $puppetdb_url,
  }
  contain puppet_enterprise::trapperkeeper::workflow

  file { $webserver_conf:
    * => $common_file_parameters,
  }

  ### Webserver routing configuration ###
  $common_webserver_routing_settings = $common_hocon_settings + {
    type => 'hash',
    path => $webserver_conf,
  }

  pe_hocon_setting { 'workflow service webroute':
    setting => 'web-router-service."puppetlabs.pe-workflow-service.service/pe-workflow-service"',
    value   => { route => '/workflow', server => 'pe-workflow-service' },
    *       => $common_webserver_routing_settings,
  }

  pe_hocon_setting { 'pe-workflow-service metrics webroute':
    setting => 'web-router-service."puppetlabs.trapperkeeper.services.metrics.metrics-service/metrics-webservice"',
    value   => { route => '/metrics', server => 'pe-workflow-service' },
    *       => $common_webserver_routing_settings,
  }

  pe_hocon_setting { 'workflow service status webroute':
    setting => 'web-router-service."puppetlabs.trapperkeeper.services.status.status-service/status-service"',
    value   => { route => '/status', server => 'pe-workflow-service' },
    *       => $common_webserver_routing_settings,
  }

  ### Webserver settings in webserver.conf ###
  puppet_enterprise::trapperkeeper::webserver_settings { $service_name:
    container          => $container,
    ssl_listen_address => $ssl_listen_address,
    ssl_listen_port    => $ssl_listen_port,
    ssl_cert           => $local_ssl_cert,
    ssl_key            => $local_ssl_key,
    notify             => Service[$service_name],
    require            => Package[$package_name],
    access_log_config  => "${etc_dir}/request-logging.xml",
    client_auth        => 'want',
  }

  ### Common trapperkeeper services ###
  $trapperkeeper_services = {
    'jetty10-service'          => 'puppetlabs.trapperkeeper.services.webserver.jetty10-service',
    'remote-activity-reporter' => 'puppetlabs.rbac-client.services.activity',
    'webrouting-service'       => 'puppetlabs.trapperkeeper.services.webrouting.webrouting-service',
    'metrics-service'          => 'puppetlabs.trapperkeeper.services.metrics.metrics-service',
    'metrics-webservice'       => 'puppetlabs.trapperkeeper.services.metrics.metrics-service',
    'scheduler-service'        => 'puppetlabs.trapperkeeper.services.scheduler.scheduler-service',
    'status-service'           => 'puppetlabs.trapperkeeper.services.status.status-service',
  }

  $trapperkeeper_services.each |$service, $namespace| {
    puppet_enterprise::trapperkeeper::bootstrap_cfg { "${container}: ${service}":
      container => $container,
      namespace => $namespace,
      service   => $service,
      notify    => Service[$service_name],
      require   => Package[$package_name],
    }
  }

  ### Service definition ###
  puppet_enterprise::trapperkeeper::init_defaults { $container:
    user                 => $user,
    group                => $group,
    service_stop_retries => $service_stop_retries,
    start_timeout        => $service_start_timeout,
    notify               => Service[$service_name],
    require              => Package[$package_name],
  }

  $_service_ensure = $disable_service ? { false => 'running', true => 'stopped' }
  puppet_enterprise::trapperkeeper::pe_service { $container:
    ensure  => $_service_ensure,
    enable  => !$disable_service,
    require => Package[$package_name],
  }

  ### Java args ###
  if $flight_recorder_enabled {
    $_jfr_java_args = puppet_enterprise::jfr_java_args(
      $flight_recorder_repository,
      $flight_recorder_filename,
      $flight_recorder_settings,
      $flight_recorder_stackdepth,
      $flight_recorder_maxage,
      $flight_recorder_maxsize
    )
  } else {
    $_jfr_java_args = {}
  }
  $_java_args = pe_merge($java_args, $_jfr_java_args)
  puppet_enterprise::trapperkeeper::java_args { $container:
    java_args                    => $_java_args,
    disable_string_deduplication => $disable_string_deduplication,
    enable_gc_logging            => $enable_gc_logging,
    notify                       => Service[$service_name],
    require                      => Package[$package_name],
  }
}
