define puppet_enterprise::trapperkeeper::java_args (
  Hash   $java_args,
  String $java_version = $puppet_enterprise::params::java_version,
  String $container   = $title,
  String $initconfdir = $puppet_enterprise::params::defaults_dir,
  Boolean $enable_gc_logging = true,
  Boolean $disable_string_deduplication = false,
  String $jdk_security_policy = $puppet_enterprise::params::jdk_security_policy,
  Integer $jdk_ephemeral_dh_key_size = $puppet_enterprise::jdk_ephemeral_dh_key_size,
  Variant[Integer[4,4], Integer[6,6]] $ip_version = $puppet_enterprise::params::ip_version,
){

  $pe_container = "pe-${container}"
  $initconf = "${initconfdir}/${pe_container}"
  $dh_key_size_args = {'Djdk.tls.ephemeralDHKeySize=' => String($jdk_ephemeral_dh_key_size)}

  # When a user decides to use a different garbage collector than the default G1, we should not
  # add the option to turn on string deduplication.
  $user_non_g1gc_present = $java_args and $java_args.any |$arg| { $arg[0] =~ /XX:\+Use(?!G1).*GC/ }

  if $disable_string_deduplication or $user_non_g1gc_present {
    $string_dedup = {}
  } else {
    $string_dedup = { 'XX:+UseStringDeduplication' => '' }
  }

  if $enable_gc_logging {
    $java_gc_log_file = "/var/log/puppetlabs/${container}/${container}_gc.log"
    $gc_logging_args = { "Xlog:gc*:file=${java_gc_log_file}" => ':time,uptime,level,tags:filecount=16,filesize=16m', }
  } else {
    $gc_logging_args = {}
  }

  if $ip_version == 6 {
    $ipv6_args = { 'Djava.net.preferIPv6Addresses=' => 'true' }
  } else {
    $ipv6_args = {}
  }

  if $facts['fips_enabled'] and $java_args['Djava.security.egd'] {
    warning("Attempting to override '-Djava.security.egd' in FIPS mode")
  }

  # the == at the end of this property name ensures that the value is written
  # as -Djava.security.properties==/opt/puppetlabs/share/jdk17-fips-security
  # the == means that this is the *only* security policy loaded by the JVM
  $_java_args = $java_args + $ipv6_args + $gc_logging_args + $dh_key_size_args + $string_dedup + {
    'Djava.security.properties==' => $jdk_security_policy
  }

  if !pe_empty($_java_args) {
    $_java_args_override = pe_join(pe_wrap_join_keys_to_values($_java_args, '-', '', ''), ' ')

    pe_ini_setting { "${pe_container}_java_args_override" :
      path              => $initconf,
      key_val_separator => '=',
      section           => '',
      setting           => 'JAVA_ARGS',
      value             => "\"${_java_args_override}\"",
      require           => Package[$pe_container],
      notify            => Exec["${pe_container} service full restart"],
    }
  }
}
