#!/opt/puppetlabs/puppet/bin/ruby
# frozen_string_literal: true

require_relative '../files/enterprise_task_helper.rb'
require 'json'
require 'puppet'
require 'open3'

class PuppetserverCAClean < EnterpriseTaskHelper
  def task(certname:, **_kwargs)
    if certname == 'localhost'
      certname_cmd = ['/opt/puppetlabs/bin/puppet', 'config', 'print', 'certname']
      certname, = Open3.capture2e(*certname_cmd)[0].strip
    end
    # This may or may not be a symlink to the new CA location
    ca_dir = '/etc/puppetlabs/puppet/ssl/ca'

    certnames = [certname].flatten

    # Check that we have either a cert or a CSR for each given certname. If not,
    # exclude it from the call to puppetserver ca clean. This is because we want
    # to expose real errors from the command, rather than the command simply being
    # unable to find any artifacts to clean.
    certnames_to_clean = []
    certnames.each do |c|
      include_certname = false
      test_cert_cmd = ['test', '-e', "#{ca_dir}/signed/#{c}.pem"]
      _output, cert_status = Open3.capture2e(*test_cert_cmd)
      include_certname ||= cert_status.exitstatus.zero?
      unless include_certname
        test_csr_cmd = ['test', '-e', "#{ca_dir}/requests/#{c}.pem"]
        _output, csr_status = Open3.capture2e(*test_csr_cmd)
        include_certname ||= csr_status.exitstatus.zero?
      end
      certnames_to_clean << c if include_certname
    end

    if certnames_to_clean.empty?
      # We don't treat this as an error, since not finding any certs or CSRs just means they are
      # already cleaned.
      { _output: 'No valid certs or CSRs found for given certnames, so nothing to clean.' }.to_json
    else
      # Even though running these one-by-one is less efficient, this allows us to detect which cleans
      # had a problem without having to parse the output of the command, and allows us to attempt
      # to clean the rest.
      errors = []
      all_output = ''
      certnames_to_clean.each do |c|
        cmd = ['/opt/puppetlabs/bin/puppetserver', 'ca', 'clean', "--certname=#{c}"]
        output, status = Open3.capture2e(*cmd)
        # 'puppetserver ca clean' returns 0 if a signed cert was cleaned up or returns 24 if a CSR was cleaned up
        if status && !status.exitstatus.zero? && status.exitstatus != 24
          errors << [c, output]
        end
        all_output += "\n" + output
      end
      # 'puppetserver ca clean' returns 0 if a signed cert was cleaned up or returns 24 if a CSR was cleaned up
      if !errors.empty?
        raise EnterpriseTaskHelper::Error.new(
          'Error attempting to clean certnames from CA.',
          'puppetlabs.puppetserver-ca-clean/puppetserver-ca-clean-failed',
          'output' => all_output,
          'errored_certnames' => errors.map { |e| e[0] },
          'errors' => errors
        )
      end
      { _output: all_output }.to_json
    end
  end
end

PuppetserverCAClean.run if __FILE__ == $PROGRAM_NAME
