# Namespaced prefix function based on join_keys_to_values from puppetlabs-stdlib
# With added pre and post strings.
# https://github.com/puppetlabs/puppetlabs-stdlib/blob/master/lib/puppet/parser/functions/join_keys_to_values.rb

module Puppet::Parser::Functions
  newfunction(:pe_wrap_join_keys_to_values, :type => :rvalue, :doc => <<-EOS
This function joins each key of a hash to that key's corresponding value with a
separator and wraps it in the pre and post strings. Keys and values are cast to strings. The return value is an array in
which each element is one joined key/value pair.

*Examples:*

    pe_wrap_join_keys_to_values({'a'=>1,'b'=>2}, "wow ", " is ", " amaze")

Would result in: ["wow a is 1 amaze","wow b is 2 amaze"]
    EOS
  ) do |arguments|

    # Validate the number of arguments.
    if arguments.size != 4
      raise(Puppet::ParseError, "pe_wrap_join_keys_to_values(): Takes exactly four " +
            "arguments, but #{arguments.size} given.")
    end

    # Validate the first argument.
    hash = arguments[0]
    if not hash.is_a?(Hash)
      raise(TypeError, "pe_wrap_join_keys_to_values(): The first argument must be a " +
            "hash, but a #{hash.class} was given.")
    end

    # Validate the second argument.
    pre = arguments[1]
    if not pre.is_a?(String)
      raise(TypeError, "pe_wrap_join_keys_to_values(): The second argument must be a " +
            "string, but a #{pre.class} was given.")
    end


    # Validate the second argument.
    separator = arguments[2]
    if not separator.is_a?(String)
      raise(TypeError, "pe_wrap_join_keys_to_values(): The third argument must be a " +
            "string, but a #{separator.class} was given.")
    end

    # Validate the third argument.
    post = arguments[3]
    if not post.is_a?(String)
      raise(TypeError, "pe_wrap_join_keys_to_values(): The second argument must be a " +
            "string, but a #{post.class} was given.")
    end

    # Join the keys to their values.
    hash.map do |k,v|
      pre + String(k) + separator + String(v) + post
    end

  end
end

# vim: set ts=2 sw=2 et :
