# This class configures the pe-patching-service.
#
# @param certname [String] The name of the patching service SSL certificate.
# @param service_name [String] The name of the service
# @param database_host [String] The hostname of the database for this service.
# @param database_port [Port] The port that the PostgreSQL database host for the
# patching service is listening on.
# @param database_password [Optional[String]] The password to use when using password
# auth on the patching service database.
# @param database_properties [String] Additional properties to add on to the
# "database.subname" value
# @param disable_service [Boolean] Whether the service should be running or not.
# By default the patching_service is disabled. Setting this to true in the PE Master
# node group allows this service to start on the primary. It is always set false on
# the replica.
# @param service_stop_retries [Integer] Number of seconds to wait during shutdown after
# the TERM signal is sent before sending a KILL signal to the process.
# @param service_start_timeout [Integer] Number of seconds to wait during startup for
# the service to finish booting before timing out.
# @param ssl_listen_address [String] The network interface address used for SSL
# connections to the service.
# @param ssl_listen_port [Integer] The port used for SSL connections to the service.
# @param puppetdb_host Array[String] Array of the hosts running puppetdb.
# @param puppetdb_port Array[Integer] Array of the ports that puppetdb is listening on
# matching the puppetdb_host array.
# @param java_args [Hash] Java args to use for the service. Defaults to setting the
# heap size.
# @param flight_recorder_enabled [Boolean] Enable/disable Java Flight Recorder for
# all Java services
# @param flight_recorder_settings [String] Settings value for JFR. The 'default'
# setting is low overhead, 'profile' gives more data.
# @param flight_recorder_stackdepth [Integer] Number of stack frames to capture.
# Default is 128 to account for Clojure/JRuby's deeply nested function calls.
# @param flight_recorder_maxage [String] Maximum amount of time to keep recorded data
# for each service. Can be specified by s (seconds), m (minutes), h (hours), or
# d (days) suffixes.
# @param flight_recorder_maxsize [String] Maximum size of recorded data files for
# each service. Can be specified with k (kilobytes), m (megabytes) and g (gigabytes)
# suffixes.
# @param flight_recorder_repository [String] Directory to save JFR chunks while recording.
# @param flight_recorder_filename [String] File to save completed recording.
# @param disable_string_deduplication [Boolean] When using the default G1 garbage
# collector, this disables the use of string deduplication for optimization.
# @param enable_gc_logging [Boolean] Enable the garbage collector log.
# @param operations_purge_run_interval_ms [Integer] The number of milliseconds between
# “purge old operations”. Default is once a day.
# @param operations_purge_age_days [Integer] The number of days old an operation needs
# to be before it is purged. Default is 60 days.
# @param node_group_enforcement_interval_seconds [Integer] The number of seconds between
# enforcement checks for the patching node-group-tree. Default is once an hour.
# @param package_updates_run_interval_ms [Integer] The number of milliseconds between
# package count updates. Default is 15 minutes.
# @param package_updates_age_minutes [Integer] The number of minutes old the package
# update must be before it is considered stale. Default is 30 minutes.
# @param operation_request_timeout [Integer] The number of seconds to wait for a new
# result on the operations/<id> endpoint to implement long polling.  Default is 60 seconds (1 minute).
# @param worker_thread_pool_termination_max [Integer] The number of seconds to wait for the
# thread pool termination at shutdown. Default is 30 seconds.
# @param max_wait_iterations [Integer] Number of iterations to wait in operations for
# various states. Default is 1000 (roughly 17 minutes given the default 1 second wait between iterations).
# @param iteration_sleep_time_ms [Integer] Sleep time in milliseconds to wait between
# iterations. Default is 1000 (1 second).
# @param connectivity_wait_iterations [Integer] The number of iterations during patch
# runs to wait when the connections can’t be established. Default is max-wait-iterations.
# @param connectivity_sleep_time_ms [Integer] The number of milliseconds to sleep between
# iterations trying to connect to a given host during patching operations.
# @param job_running_wait_iterations [Integer] The number of iterations to wait for the
# given patch job to complete once the connection to the orchestrator is established.
# Default is $max_wait_iterations.
# @param job_running_sleep_time_ms [Integer] The amount of time in milliseconds to sleep
# between iterations waiting for the job to complete. Default is $iteration_sleep_time_ms.
# @param bootstrapping_check_interval [Integer] The amount of time in seconds to sleep
# between attempts to bootstrap the node groups on first startup of the system after
# activation. Default is 2 seconds.
# @ param bootstrapping_check_iterations [Integer] The number of retries to successfully
# bootstrap the patch groups on first activation.  Defaults to 15.
class puppet_enterprise::profile::patching_service (
  String $certname                                 = $facts['clientcert'],
  String $service_name                             = 'pe-patching-service',
  String $database_host                            = $puppet_enterprise::patching_database_host,
  Variant[String, Integer] $database_port          = $puppet_enterprise::database_port,
  Optional[String] $database_password              = $puppet_enterprise::patching_database_password,
  String $database_properties                      = $puppet_enterprise::database_properties,
  Boolean $disable_service                         = true,
  Integer $service_stop_retries                    = 60,
  Integer $service_start_timeout                   = 300,
  String $ssl_listen_address                       = $puppet_enterprise::params::ssl_address,
  Integer $ssl_listen_port                         = $puppet_enterprise::patching_service_port,
  Array[String]  $puppetdb_host                    = $puppet_enterprise::puppetdb_hosts_array,
  Array[Integer] $puppetdb_port                    = $puppet_enterprise::puppetdb_ports_array,
  Hash $java_args                                  = $puppet_enterprise::params::patching_service_java_args,
  Boolean $flight_recorder_enabled                 = $puppet_enterprise::flight_recorder_enabled,
  String $flight_recorder_settings                 = $puppet_enterprise::flight_recorder_settings,
  Integer $flight_recorder_stackdepth              = $puppet_enterprise::flight_recorder_stackdepth,
  String $flight_recorder_maxage                   = $puppet_enterprise::flight_recorder_maxage,
  String $flight_recorder_maxsize                  = $puppet_enterprise::flight_recorder_maxsize,
  String $flight_recorder_repository               = '/var/log/puppetlabs/patching-service/jfr',
  String $flight_recorder_filename                 = '/var/log/puppetlabs/patching-service/last_recording.jfr',
  Boolean $disable_string_deduplication            = $puppet_enterprise::disable_string_deduplication,
  Boolean $enable_gc_logging                       = true,
  Integer $operations_purge_run_interval_ms        = 86400000,
  Integer $operations_purge_age_days               = 60,
  Integer $node_group_enforcement_interval_seconds = 3600,
  Integer $package_updates_run_interval_ms         = 900000,
  Integer $package_updates_age_minutes             = 30,
  Integer $operation_request_timeout               = 60,
  Integer $worker_thread_pool_termination_max      = 30,
  Integer $max_wait_iterations                     = 1000,
  Integer $iteration_sleep_time_ms                 = 1000,
  Integer $connectivity_wait_iterations            = $max_wait_iterations,
  Integer $connectivity_sleep_time_ms              = $iteration_sleep_time_ms,
  Integer $job_running_wait_iterations             = $max_wait_iterations,
  Integer $job_running_sleep_time_ms               = $iteration_sleep_time_ms,
  Integer $bootstrapping_check_interval            = 2,
  Integer $bootstrapping_check_iterations          = 15,
) inherits puppet_enterprise {
  $container = 'patching-service'
  $package_name = 'pe-patching-service'
  $user = 'pe-patching-service'
  $group = 'pe-patching-service'
  $etc_dir = "/etc/puppetlabs/${container}"
  $confdir  = "${etc_dir}/conf.d"
  $webserver_conf = "${confdir}/webserver.conf"
  $ssl_dir = "${puppet_enterprise::server_data_dir}/patching-service/ssl"
  $local_ssl_cert = "${ssl_dir}/${certname}.cert.pem"
  $local_ssl_key = "${ssl_dir}/${certname}.private_key.pem"
  $puppetdb_url = "https://${puppetdb_host[0]}:${puppetdb_port[0]}"
  if $puppet_enterprise::database_ssl and $puppet_enterprise::database_cert_auth {
    $client_pk8_key  = "${ssl_dir}/${certname}.private_key.pk8"
    $client_cert     = "${ssl_dir}/${certname}.cert.pem"
    $ssl_database_properties = "${database_properties}&sslkey=${client_pk8_key}&sslcert=${client_cert}"
  } else {
    $ssl_database_properties = $database_properties
  }

  $common_hocon_settings = {
    ensure  => present,
    notify  => Service[$service_name],
    require => Package[$package_name],
  }

  $common_file_parameters = {
    ensure  => file,
    owner   => $user,
    group   => $group,
    mode    => '0640',
    require => Package[$package_name],
  }

  ### Package ###
  include puppet_enterprise::packages
  Package <| tag == 'pe-patching-packages' |> {
    before => [Class[puppet_enterprise::trapperkeeper::patching]],
  }

  ### Make copy of cert/key in service ssl directory ###
  puppet_enterprise::certs { 'pe-patching-service::ssl':
    certname      => $certname,
    container     => $container,
    cert_dir      => $ssl_dir,
    append_ca     => false,
    make_pk8_cert => true,
    notify        => Service[$service_name],
    require       => Package[$package_name],
  }

  ### global.conf ###
  puppet_enterprise::trapperkeeper::global_settings { $container :
    certname    => $certname,
    ssl_cert    => $local_ssl_cert,
    ssl_key     => $local_ssl_key,
    localcacert => $puppet_enterprise::params::localcacert,
    require     => Package[$package_name],
  }

  ### analytics client settings ###
  puppet_enterprise::trapperkeeper::analytics_client_settings { $container:
    container => $container,
    owner     => $user,
    notify    => Service[$service_name],
    require   => Package[$package_name],
  }

  ### Patching service configuration ###
  class { 'puppet_enterprise::trapperkeeper::patching':
    service_name                            => $service_name,
    database_host                           => $database_host,
    database_port                           => $database_port,
    database_password                       => $database_password,
    database_properties                     => $ssl_database_properties,
    puppetdb_url                            => $puppetdb_url,
    operations_purge_run_interval_ms        => $operations_purge_run_interval_ms,
    operations_purge_age_days               => $operations_purge_age_days,
    node_group_enforcement_interval_seconds => $node_group_enforcement_interval_seconds,
    package_updates_run_interval_ms         => $package_updates_run_interval_ms,
    package_updates_age_minutes             => $package_updates_age_minutes,
    operation_request_timeout               => $operation_request_timeout,
    worker_thread_pool_termination_max      => $worker_thread_pool_termination_max,
    max_wait_iterations                     => $max_wait_iterations,
    iteration_sleep_time_ms                 => $iteration_sleep_time_ms,
    connectivity_wait_iterations            => $connectivity_wait_iterations,
    connectivity_sleep_time_ms              => $connectivity_sleep_time_ms,
    job_running_wait_iterations             => $job_running_wait_iterations,
    job_running_sleep_time_ms               => $job_running_sleep_time_ms,
    bootstrapping_check_interval            => $bootstrapping_check_interval,
    bootstrapping_check_iterations          => $bootstrapping_check_iterations,
  }
  contain puppet_enterprise::trapperkeeper::patching

  file { $webserver_conf:
    * => $common_file_parameters,
  }

  ### Webserver routing configuration ###
  $common_webserver_routing_settings = $common_hocon_settings + {
    type => 'hash',
    path => $webserver_conf,
  }

  pe_hocon_setting { 'patching service webroute':
    setting => 'web-router-service."puppetlabs.pe-patching-service.service/pe-patching-service"',
    value => { route => '/patching', server => 'pe-patching-service' },
    *     => $common_webserver_routing_settings,
  }

  pe_hocon_setting { 'vulnerability service webroute':
    setting => 'web-router-service."puppetlabs.vulnerability-remediation.service/vulnerability-remediation-service"',
    value => { route => '/vrs', server => 'pe-patching-service' },
    *     => $common_webserver_routing_settings,
  }

  pe_hocon_setting { 'pe-patching-service metrics webroute':
    setting => 'web-router-service."puppetlabs.trapperkeeper.services.metrics.metrics-service/metrics-webservice"',
    value   => { route => '/metrics', server => 'pe-patching-service' },
    *       => $common_webserver_routing_settings,
  }

  pe_hocon_setting { 'patching service status webroute':
    setting => 'web-router-service."puppetlabs.trapperkeeper.services.status.status-service/status-service"',
    value   => { route => '/status', server => 'pe-patching-service' },
    *       => $common_webserver_routing_settings,
  }

  pe_hocon_setting { 'patching service callback webroute':
      setting => 'web-router-service."puppetlabs.trapperkeeper.services.callback.callback-service/callback-service"',
      value   => { route => '/callback', server => 'pe-patching-service' },
      *       => $common_webserver_routing_settings,
    }

  ### Webserver settings in webserver.conf ###
  puppet_enterprise::trapperkeeper::webserver_settings { $service_name:
    container          => $container,
    ssl_listen_address => $ssl_listen_address,
    ssl_listen_port    => $ssl_listen_port,
    ssl_cert           => $local_ssl_cert,
    ssl_key            => $local_ssl_key,
    notify             => Service[$service_name],
    require            => Package[$package_name],
    access_log_config  => "${etc_dir}/request-logging.xml",
    client_auth        => 'want',
  }

  ### Common trapperkeeper services ###
  $trapperkeeper_services = {
    'jetty10-service'          => 'puppetlabs.trapperkeeper.services.webserver.jetty10-service',
    'remote-activity-reporter' => 'puppetlabs.rbac-client.services.activity',
    'webrouting-service'       => 'puppetlabs.trapperkeeper.services.webrouting.webrouting-service',
    'metrics-service'          => 'puppetlabs.trapperkeeper.services.metrics.metrics-service',
    'metrics-webservice'       => 'puppetlabs.trapperkeeper.services.metrics.metrics-service',
    'scheduler-service'        => 'puppetlabs.trapperkeeper.services.scheduler.scheduler-service',
    'status-service'           => 'puppetlabs.trapperkeeper.services.status.status-service',
    'callback-service'         => 'puppetlabs.trapperkeeper.services.callback.callback-service',
  }

  $trapperkeeper_services.each |$service, $namespace| {
    puppet_enterprise::trapperkeeper::bootstrap_cfg { "${container}: ${service}":
      container => $container,
      namespace => $namespace,
      service   => $service,
      notify    => Service[$service_name],
      require   => Package[$package_name],
    }
  }

  ### Service definition ###
  puppet_enterprise::trapperkeeper::init_defaults { $container:
    user                 => $user,
    group                => $group,
    service_stop_retries => $service_stop_retries,
    start_timeout        => $service_start_timeout,
    notify               => Service[$service_name],
    require              => Package[$package_name],
  }

  $_service_ensure = $disable_service ? { false => 'running', true => 'stopped' }
  puppet_enterprise::trapperkeeper::pe_service { $container:
    ensure  => $_service_ensure,
    enable  => !$disable_service,
    require => Package['pe-patching-service'],
  }

  ### Java args ###
  if $flight_recorder_enabled {
    $_jfr_java_args = puppet_enterprise::jfr_java_args(
      $flight_recorder_repository,
      $flight_recorder_filename,
      $flight_recorder_settings,
      $flight_recorder_stackdepth,
      $flight_recorder_maxage,
      $flight_recorder_maxsize
    )
  } else {
    $_jfr_java_args = {}
  }
  $_java_args = pe_merge($java_args, $_jfr_java_args)
  puppet_enterprise::trapperkeeper::java_args { $container:
    java_args                    => $_java_args,
    disable_string_deduplication => $disable_string_deduplication,
    enable_gc_logging            => $enable_gc_logging,
    notify                       => Service[$service_name],
    require                      => Package[$package_name],
  }
}
