#!/usr/bin/env ruby
# frozen_string_literal: true

# @!visibility private
# @summary Modifies configuration on a NETCONF device
#
# This task demonstrates the NETCONF edit-config operation for making configuration
# changes. It supports the standard NETCONF workflow of lock, edit, commit, unlock
# when working with the candidate datastore, or direct edits to the running datastore.
#
# @example Edit candidate configuration with automatic commit
#   bolt task run netconf::edit_config config='<config>...</config>' target=candidate --targets router1
#
# @example Direct edit to running configuration
#   bolt task run netconf::edit_config config='<config>...</config>' target=running --targets router1
#
# @example Dry run to preview changes
#   bolt task run netconf::edit_config config='<config>...</config>' --noop --targets router1
#
# @param config [String] Configuration to apply as an XML string
#   Must be a valid XML configuration fragment appropriate for the target device.
#
# @param target [String] Target datastore ('running' or 'candidate')
#   Defaults to 'candidate'. When using 'candidate', changes are automatically committed.
#
# @param operation [String] NETCONF operation ('merge', 'replace', or 'none')
#   Defaults to 'merge'. Controls how the configuration is applied.
#
# @param target_datastore [String] Override the default target datastore for the session
#   Allows specifying a different default datastore for all operations in the session.
#
# @param redact_patterns [Array<String>] Patterns to redact from logs and output
#   Regular expressions that match sensitive data to be redacted.
#
# @return [Hash] Task result containing:
#   - 'target' [String]: The datastore that was modified
#   - 'operation' [String]: The operation that was performed
#   - 'committed' [Boolean]: Whether changes were committed (for candidate datastore)
#   - 'session_report' [Hash]: Detailed session information including change tracking
#   - config: XML configuration to apply (required)
#   - target: Target datastore (default: 'candidate')
#   - commit: Whether to commit after edit (default: true)

# Include the task helper to set up load paths
require_relative '../files/task_helper.rb'

# Now require the netconf library
require 'puppet_x/puppetlabs/netconf/session'
require 'json'

result = PuppetX::Puppetlabs::Netconf::Session.with_session do |session|
  # 1. User: wants to manage VLANS on device1
  # 2. AI: tool: get_device_info -> list capabilities -> get VLAN schema
  # 3. AI: Here is a task that can create VLANS -> prints out the Bolt task
  # 4. User: Make the task support device2 now. it's running different firmware.

  # Get task parameters
  config_xml = session.task_params['config']
  # NOTE: target_datastore is already set in the session from the parameter
  # The session uses @target_datastore internally
  commit = session.task_params.fetch('commit', true)

  unless config_xml
    raise "Missing required parameter 'config' - must provide XML configuration to apply"
  end

  # Get the target from the session (it was set from target_datastore parameter)
  target = session.instance_variable_get(:@target_datastore) || 'candidate'
  session.logger.info("Editing #{target} configuration")

  # Apply the configuration - edit_config doesn't take target as parameter
  session.edit_config(config_xml)
  session.logger.info('Configuration edit successful')

  # Commit if requested and target is candidate
  if commit && target == 'candidate'
    session.logger.info('Committing configuration changes')
    session.commit
    session.logger.info('Configuration committed successfully')
  end

  # Report success
  session.report_result({
                          'status' => 'success',
    'target' => target,
    'committed' => commit && target == 'candidate',
    'message' => "Successfully edited #{target} configuration"
                        })
end

# Output the result as JSON for Bolt
puts result.to_json
