# The pe_install class coordinates all of the puppet_enterprise profiles
# necessary to bring a node into a state matching the designated roles based on
# certname.
#
# Typically this is the only class that needs to be included. All other classes
# are internal. The `puppet infrastructure` face in the pe_manager module will
# include it automatically when configuring a node.
#
# See the [README](./README.md) for more details.
#
# @param ca_certname [String] the certname for the node on which
#   certificate_authority services should be installed
# @param master_certname [String] the certname for the node on which
#   puppetserver and orchestrator services should be installed
# @param puppetdb_certname [String] the certname for the node on which
#   the puppetdb service should be installed
# @param console_certname [String] the certname for the node on which the
#   console services should be installed
# @param database_certname [String] the certname for the node on which the
#   postgresql database services should be installed
# @param puppet_master_dnsaltnames [Array[String]] a list of subjectaltnames to
#   be included in the primary's certificate (in addition to cn).
# @param enforce_pe_master_rule [Boolean] When true, enforces the 
#   trusted.extensions.pp_auth_role = pe_compiler rule on PE Master. This should
#   not be enforced on upgrades or if no new-style PE Compilers exist, which is
#   determined by puppet infra configure.
# @param console_admin_password [String] the password for the admin user of the
#   PE console
# @param signing_ca [Struct] a Struct containing file paths for a CA that will be
#   imported in as the CA that puppetserver ca will use for all of its operations.
#   The key/value pairs required by the Struct are detailed below:
#     * bundle - All of the CA certifciates concatenated together in their order
#                of authority, with the lowest authority first.
#
#                For example, if you had a root and an intermediate CA certificate,
#                the intermediate would be added to the bundle first, the the root
#                would need to be concatendated in.
#     * crl_chain - All of the CRLs concatenated together in the order of authority,
#                with the lowest authority first. See the example from bundle.
#     * private_key - The private key for the CA cert that you are importing into
#                puppetserver. This key must be for the least authoritative cert
#                in the bundle and crl_chain files.
# @param bootstrap_to [Optional[String]] The PE version we are installing (could be
#   either a first install or an upgrade). If undef, this is a remediation
#   configure run outside of a bootstrapping install.
# @param upgrade_from [Optional[String]] The PE version we are upgrading from.
#   Used to control the inclusion of classes which are only needed during an upgrade.
# @param repairing [Optional[Boolean]] Note if the user is attempting to repair an installaion
#   Used to prevent installation of out-of-box environment groups
# @param ca_only
#   Temporarily install just the CA profile. Used as a bootstrapping step for
#   split installs to prep the CA for the database node without attempting to install
#   additional profiles that depend on the database node.
class pe_install(
  String $ca_certname               = $::puppet_enterprise::certificate_authority_host,
  String $master_certname           = $::puppet_enterprise::puppet_master_host,
  Variant[String,Array[String]] $puppetdb_certname = $::puppet_enterprise::puppetdb_host,
  String $console_certname          = $::puppet_enterprise::console_host,
  String $database_certname         = $::puppet_enterprise::database_host,
  Array[String] $puppet_master_dnsaltnames = ['puppet'],
  Boolean $enforce_pe_master_rule   = false,
  Optional[String] $console_admin_password = lookup('console_admin_password', {'default_value' => undef }),
  Optional[Struct[{
    bundle => String[1],
    crl_chain => String[1],
    private_key => String[1],
  }]] $signing_ca                   = undef,
  Optional[String] $bootstrap_to    = undef,
  Optional[String] $upgrade_from    = undef,
  Optional[Boolean] $repairing      = false,
  Boolean $ca_only               = false,
) inherits ::puppet_enterprise {

  $is_ca       =  $ca_certname ==  $facts['clientcert']
  $is_master   =  $master_certname ==  $facts['clientcert']
  $is_orchestrator = ($is_master and !$ca_only)
  $is_puppetdb =  (($facts['clientcert'] in pe_flatten([$puppetdb_certname])) and !$ca_only)
  $is_console  =  (($console_certname == $facts['clientcert']) and !$ca_only)

  $is_install  =  !pe_empty($bootstrap_to)
  $is_upgrade  =  !pe_empty($upgrade_from)

  $puppetdb_database_host = $::puppet_enterprise::puppetdb_database_host
  $console_database_host  = $::puppet_enterprise::console_database_host
  $orchestrator_database_host = $::puppet_enterprise::orchestrator_database_host

  $is_database = (($facts['clientcert'] in [ $database_certname,
                                    $puppetdb_database_host,
                                    $console_database_host,
                                    $orchestrator_database_host ]) and ! $ca_only)
  $puppet_service_managed = lookup('pe_infrastructure::agent::puppet_service_managed', { 'default_value' => true })

  class { '::pe_install::validate': }
  -> class { '::pe_install::prepare': }
  -> class { '::pe_install::install': }

  contain '::pe_install::validate'
  contain '::pe_install::prepare'
  contain '::pe_install::install'
}
