# Stops and disables all the pe services prior to upgrading the packages.
#
# @param force_service_halt [Variant[Boolean,Enum['by_version']]] flag for
#   determining whether services are shutdown before packages are upgraded.
#   By default, is determined by whether an upgrade version was supplied
#   to the pe_install class.
class pe_install::upgrade::stop_services(
  Boolean $force_service_halt = $::pe_install::is_upgrade,
) {
  include 'pe_install::params'
  include 'puppet_enterprise::packages'

  if $force_service_halt {
    $_pg_before_refs = $::pe_install::is_database ? {
      true => [
        Package['postgresql-client'],
        Package['postgresql-server'],
        Package['postgresql-contrib'],
      ],
      false => []
    }
    $_pg_remove_before_refs = $::pe_install::is_database ? {
      true  => [
        Class['puppet_enterprise::postgresql::remove'],
      ],
      false =>  []
    }
    $_console_before_refs = $::pe_install::is_console ? {
      true => [
        Package['pe-nginx'],
      ],
      false => []
    }

    $before_refs = [Class['Puppet_enterprise::Packages']] + $_pg_before_refs + $_pg_remove_before_refs + $_console_before_refs

    if $::pe_install::is_console {
      [
        'pe-nginx',
        'pe-console-services',
      ].each |$service| {
        pe_install::stop_service { "Stop ${service} service":
          service          => $service,
          before_reference => $before_refs,
        }
      }
    }
    if $::pe_install::is_puppetdb {
      [
        'pe-puppetdb',
      ].each |$service| {
        pe_install::stop_service { "Stop ${service} service":
          service          => $service,
          before_reference => $before_refs,
        }
      }
    }
    if $::pe_install::is_database {
      [
        'pe_databases-catalogs.timer', 'pe_databases-facts.timer', 'pe_databases-other.timer', 'pe_databases-reports.timer', 'pe_databases-resource_events.timer',
        'pe_databases-catalogs.service', 'pe_databases-facts.service', 'pe_databases-other.service', 'pe_databases-reports.service', 'pe_databases-resource_events.service',
      ].each |$database_service| {
        pe_install::stop_service { "Stop ${database_service} service":
          service  => $database_service,
          before_reference => $before_refs,
        }
      }
      [
        'pe-postgresql',
      ].each |$service| {
        # PE-26482: Workaround for systemd 237 bug
        $disable_service = ($facts['os']['name'] != 'Ubuntu') or ($facts['os']['release']['major'] !~ /18.04/)
        $postgres_dir = "${pe_install::params::server_data_dir}/postgresql"
        $pe_postgresql_info = $facts['pe_postgresql_info']
        if empty($pe_postgresql_info) {
          fail('pe_postgresql_info fact is missing, unable to determine pe-postgresql server version')
        }
        $installed_server_version = $pe_postgresql_info['installed_server_version']
        $pe_postgresql_pid_file = "${postgres_dir}/${installed_server_version}/data/postmaster.pid"
        pe_install::stop_service { "Stop ${service} service":
          service          => $service,
          before_reference => $before_refs,
          disable_service  => $disable_service,
          stop_command     => "kill -INT `head -1 ${pe_postgresql_pid_file}` && while [ -e ${pe_postgresql_pid_file} ]; do sleep 1; done"
        }
      }
    }
    if $::pe_install::is_master {

      if versioncmp(pe_current_server_version(), '2015.3.0') >= 0 {
        pe_install::stop_service { 'Stop pe-orchestration-services service':
          service          => 'pe-orchestration-services',
          before_reference => $before_refs,
        }
      }

      [
        'pe-puppetserver',
        'pe-bolt-server',
        'pe-ace-server',
        'pe-host-action-collector',
        'pe-patching-service',
        'pe-infra-assistant',
        'pe-workflow-service',
      ].each |$service| {
        pe_install::stop_service { "Stop ${service} service":
          service          => $service,
          before_reference => $before_refs,
        }
      }
    }
  }
}
