require 'puppet/application/face_base'
require 'puppet_x/puppetlabs/meep/infra/feature_flags'

module PuppetX
  module Util
    class InfrastructureHelp
      extend PuppetX::Puppetlabs::Meep::Infra::FeatureFlags

      def self.action_help(action)
        case action
        when :run
          run_help
        when :configure
          configure_help
        when :enable
          enable_help
        when :forget
          forget_help
        when :promote
          promote_help
        when :provision
          provision_help
        when :recover_configuration
          recover_configuration_help
        when :reinitialize
          reinitialize_help
        when :status
          status_help
        when :tune
          tune_help
        when :upgrade
          puts Puppet::Face[:help, :current].help('infrastructure', action) if Puppet::Face[:infrastructure, :current].actions.include?(:upgrade)
        when :uninstall
          uninstall_help
        when :console_password
          console_password_help
        else
          Puppet.err _("%{action} is not a recognized action. See `puppet infrastructure help` for available actions.") % { action: action }
        end
      end

      def self.configure_help
        _('DESCRIPTION:') + "\n" +
        PuppetX::Util::String::Formatter.join_and_wrap_no_indent(
          [_('Validates the current installation and configures the local system into a desired state described by the Puppet configuration file(s) in /etc/puppetlabs/enterprise with puppet apply.'),
          _('This action can be called with install or upgrade options which will perform install/upgrade on the current node, based on the /etc/puppetlabs/enterprise hiera data.'), 
          _('In the case of upgrades, it also recovers configuration into pe.conf from the existing PE configuration to eliminate drift since the previous install.'),
          _('It also performs validation and version lookup to determine if a postgres migration is needed and possible.')]) +

          Puppet::Face[:help, :current].help('infrastructure', :configure)

      end

      def self.enable_help
        _('DESCRIPTION:') + "\n" +
        PuppetX::Util::String::Formatter.wrap_no_indent(
          _('Enables a provisioned replica to take over from a primary during a failover.') +
          _('This is done by updating classification such that all components of your PE
          installation will know to contact the replica in the case of a failure.')) + "\n\n"

        _('The enable interview asks you to describe your installation:') + "\n" +

        PuppetX::Util::String::Formatter.two_column_table(_('mono'),_('All PE services run on the primary and all agents connect to that primary.')) + "\n" +
        PuppetX::Util::String::Formatter.two_column_table(_('mono-with-compile'),_('All PE services run on the primary but agents connect to compilers via a load balancer. PE infrastructure nodes will still check in with the primary and fail over to the replica.')) + "\n" +

        _('You can also specify topology with the --topology flag.') + "\n\n" +

        PuppetX::Util::String::Formatter.wrap_no_indent(
          _('If mono-with-compile is selected, the load balancer hostname and port must be provided. ') + _('This is done either during the interview or with the --agent-server-urls and --pcp-broker-list flags. ') + _('You will have the opportunity to review and confirm changes before they are made.')) + "\n\n" +

        _('REQUIREMENTS:') + "\n" +

        PuppetX::Util::String::Formatter.join_and_indent(
          [_('- A provisioned replica to enable.'),
          _('- A valid token with permissions to run the Puppet agent with orchestrator. You can generate a token with `puppet access login`.')]) + "\n\n" +

        _('This command must be invoked on the primary.') +
        
        Puppet::Face[:help, :current].help('infrastructure', :enable)

      end
            
      def self.forget_help
        _('DESCRIPTION:') + "\n" +
        _('Performs required cleanup if a replica goes offline permanently.') + "\n\n" +

        PuppetX::Util::String::Formatter.format(_('IMPORTANT: You must run this command after a replica goes permanently offline, or the performance of Puppet Enterprise will degrade over time.')) + "\n\n" +

        PuppetX::Util::String::Formatter.wrap_no_indent(
          _('Forgetting a replica is required whenever a replica node is destroyed, even if you plan to replace it with a replica with the same name. ') +
          _('This command cleans up classification and database state. ') +
          _('If you fail to forget a destroyed replica, agent runs take longer and the primary\'s disk eventually fills with pending replication statements.')) + "\n\n" +

        PuppetX::Util::String::Formatter.join_and_wrap_no_indent([_('USAGE: puppet infrastructure forget NODE NAME'),'[--token-file PATH]', '[--verbose]', '[--debug]', '[--trace]' ]) + "\n\n" +

        _('REQUIREMENTS:') + "\n" +
        PuppetX::Util::String::Formatter.join_and_indent([_('- Target node must be permanently offline.'), _('- This command must be invoked on the primary.')]) +
        Puppet::Face[:help, :current].help('infrastructure', :forget)

      end

      def self.promote_help
        _('DESCRIPTION:') + "\n" +
        PuppetX::Util::String::Formatter.join_and_wrap_no_indent(
          [_('Permanently promotes a replica to be the new primary.'),
          _('Run this command after verifying that the old primary has been taken offline permanently, because any requests to the old primary during the promotion process could leave your Puppet Enterprise infrastructure in a broken state.')]) +

          Puppet::Face[:help, :current].help('infrastructure', :promote)


      end

      def self.provision_help
        help = _('DESCRIPTION:') + "\n" +
        _('Provisions a new secondary infrastructure node.') + "\n\n" +
        PuppetX::Util::String::Formatter.join_and_wrap_no_indent([
          _('Replica:'),
          _('The replica node duplicates services and components from the primary.'),
          _('The node provided must be a Puppet agent, but not another Puppet Enteprise infrastructure node. ') +
          _('For example, you cannot provide an existing compiler as an argument to this command.')]) + "\n\n" +

        PuppetX::Util::String::Formatter.join_and_wrap_no_indent(
          [_('You can create a single replica of your primary.'),
          _('If you already have a replica and wish to replace it, take the old replica offline and run `puppet infrastructure forget <NODE NAME>`.'),
          _('You can then provision a new replica.')]) + "\n\n"

        help += PuppetX::Util::String::Formatter.join_and_wrap_no_indent(
          [_('Compiler:'),
          _('The new compiler node will have both the Master and PuppetDB profiles applied.'),
          _('The node provided may be any node reachable by the primary that is not an existing infrastructure node.')]) + "\n\n"

        help += _('REQUIREMENTS:') + "\n" +
        PuppetX::Util::String::Formatter.join_and_indent(
          [_('- An agent node to provision the replica or compiler on, running the same platform as your primary. It must not already be an infrastructure node.'),
          _('- You must use the Code Manager workflow to deploy your code. Code Manager must be running and you must have deployed code at least once.'),
          _('- A valid token that has permission to run the Puppet agent with orchestrator. You can generate a token with `puppet access login`.'), 
          _('- This command must be invoked on the primary.')]) + "\n\n" +

        Puppet::Face[:help, :current].help('infrastructure', :provision)
        help
      end

      def self.recover_configuration_help
        _('DESCRIPTION:') + "\n" +
        PuppetX::Util::String::Formatter.wrap_no_indent(
          _('Run on infrastructure node to recover configuration set outside of pe.conf
          through PE Classifier or Hiera into pe.conf and node specific configuration files.')) +

          Puppet::Face[:help, :current].help('infrastructure', :recover_configuration)

      end

      def self.reinitialize_help
        _('DESCRIPTION:') + "\n" +
        PuppetX::Util::String::Formatter.wrap_no_indent(
          _('Run on replica node to clear the replicated database and resynchronize with the primary')) +

          Puppet::Face[:help, :current].help('infrastructure', :reinitialize)

      end

      def self.status_help
        _('DESCRIPTION:') + "\n" +
        PuppetX::Util::String::Formatter.wrap_no_indent(
          _('Displays error states or any other alerts from Puppet Enterprise services. ') +
          _('Services in the list are ordered by node. ') +
          _('Any alerts are listed below the corresponding service.')) + "\n\n" +

        PuppetX::Util::String::Formatter.wrap_no_indent(
        _('If you expect a node to be listed and its not shown, or the status command is trying to contact a node that you\'ve removed, run Puppet on the primary again to update the status command\'s configuration.')) + "\n" +

        Puppet::Face[:help, :current].help('infrastructure', :status)

      end

      def self.tune_help
        _('DESCRIPTION:') + "\n" +
        PuppetX::Util::String::Formatter.wrap_no_indent(
          _('Collects information about your Puppet Enterprise installation. ') +
          _('Outputs current or optimized settings that tune Puppet Enterprise services.')) +

        Puppet::Face[:help, :current].help('infrastructure', :tune)
      end

      def self.run_help
        _('DESCRIPTION:') + "\n" +
        PuppetX::Util::String::Formatter.wrap_no_indent(
          _('Run various Puppet Bolt plans through the puppet infrastructure command line interface. ') +
          _('Enables a variety of maintenance commands to be run within PE.')) +        
          
          Puppet::Face[:help, :current].help('infrastructure', :run)

      end

      def self.uninstall_help
        _('Not yet supported.')
      end
      
      def self.console_password_help
        _('DESCRIPTION:') + "\n" +
        PuppetX::Util::String::Formatter.wrap_no_indent(
          _('Changes the password for the console \'admin\' user. ') + 
          _('If a password is not supplied, the user is prompted to enter one.')) +

        Puppet::Face[:help, :current].help('infrastructure', :console_password)

      end
    end
  end
end
