# == Class: puppet_enterprise::packages
#
# This class contains virtual resources for all packages managed by PE.
# This is necessary as a single node may contain multiple profiles
# where each profile could want to install the same package, e.g.
# pe-java, causing a compilation error.
#
# @param installing [String] Defaults to false. Set true by the installer to bypass
#   modifying the local repository configuration.
#
# === Examples
#
# include puppet_enterprise::packages
#
# Package <| tag == 'master' |>
#
# @param installing [Boolean] The installer will override this if we are in the
#   midst of installing and don't want to manage the locally configured
#   repository configuration yet.
# @pe_ver [String] Normally this is either the pe_build fact from the node we are compiling
#   for, or if that's not present, the pe_build version on the primary. For certain install
#   cases (replica migration for example) this may be overridden by the installer to get the
#   correct package repository configuration even though pe_build has not be updated on the node
#   yet.
class puppet_enterprise::packages(
  Boolean $installing = false,
  String $pe_ver = pe_empty($facts['pe_build']) ? { false => $facts['pe_build'], true => pe_build_version() },
){
  include puppet_enterprise::params

  # When installing PE (first time or upgrading), we should rely on the local
  # packages repo instead of the primary's package repo link that is setup by
  # puppet_enterprise::repo. Once the MEEP run is complete, a regular
  # puppet run should complete the setup of the repo configuration pointing
  # to the primary (either local or networked). This parameter is set
  # by MEEP when building the catalog during `puppet-infra configure`.
  class { 'puppet_enterprise::repo' :
    pe_ver => $pe_ver,
    manage   => !$installing,
  }

  Package {
    ensure => latest,
    require => Class['puppet_enterprise::repo'],
    *      => $puppet_enterprise::params::package_options,
  }

  package { 'pe-bundler':
    ensure => absent,
  }

  package { 'puppet5-release':
    ensure => 'absent',
  }

  package { 'puppet6-release':
    ensure => 'absent',
  }

  package { 'puppet-release':
    ensure => 'absent',
  }

  @package { 'pe-postgresql-common':
    tag =>  [
      'pe-psql-common',
    ],
  }


  @package { $puppet_enterprise::params::postgresql_pglogical_package_name:
    tag => [
      'pe-psql-pglogical',
    ],
  }

  @package { $puppet_enterprise::params::postgresql_pgrepack_package_name:
    tag => [
      'pe-database-extensions',
    ],
  }

  @package { 'pe-java-devel':
    tag => [
      'pe-master-packages',
    ],
    ensure => absent,
    before => Package[$puppet_enterprise::params::java_package_name],
  }

  # dnf gets confused with two packages that conflict with each other
  # and obsolete the same package. Yum is fine with it for now, but is also
  # fine with this exclude package option.  dnf--
  if ($facts['os']['family'] =~ /RedHat/){
    $available_java_packages = pe_prefix($puppet_enterprise::params::available_java_versions,'pe-java')
    $excluded_java_packages = $available_java_packages.filter |$pkg| { $pkg != $puppet_enterprise::params::java_package_name }
    $pejava_install_opts = empty($excluded_java_packages) ? {
      true => undef,
      default => ['-x',pe_join($excluded_java_packages,',')],
    }
  } else {
    $pejava_install_opts = undef
  }

  @package { $puppet_enterprise::params::java_package_name:
    tag => [
      'pe-master-packages',
      'pe-puppetdb-packages',
      'pe-console-packages',
    ],
    install_options => $pejava_install_opts
  }

  @package { 'pe-bouncy-castle-jars':
    tag => [
      'pe-master-packages',
      'pe-puppetdb-packages',
      'pe-console-packages',
    ],
  }

  @package { 'pe-puppetdb':
    tag => 'pe-puppetdb-packages',
  }

  @package { 'pe-console-services':
    tag => 'pe-console-packages',
  }

  @package { 'pe-client-tools':
    tag => 'pe-controller-packages',
  }

  @package { [
    'pe-license',
    'pe-puppetdb-termini',
    'pe-console-services-termini',
    'pe-puppetserver',
    'pe-modules',
    'pe-tasks',
    'pe-backup-tools',
  ]:
    tag => 'pe-master-packages',
  }

  @package { 'pe-builtin-bolt-content':
    tag => [
      'pe-master-packages',
      'pe-orchestrator-packages',
    ],
  }

  @package { 'pe-installer':
    tag => 'pe-installer-packages',
  }

  # This should only be present on the primary/replica. It is not
  # part of pe-installer, but that's why it is tagged with pe-installer-packages.
  @package { 'pe-host-action-collector':
    tag => 'pe-installer-packages',
  }

  @package { 'pe-orchestration-services':
    tag => 'pe-orchestrator-packages',
  }

  @package { 'pe-bolt-server':
    tag => 'pe-bolt-server-packages',
    before => Package['pe-puppetserver'],
  }

  @package { 'pe-ace-server':
    tag => 'pe-ace-server-packages',
  }

  @package { 'pe-patching-service':
    tag =>  'pe-patching-packages',
  }

  @package { 'pe-infra-assistant':
    tag => 'pe-infra-assistant-packages',
  }

  @package { 'pe-workflow-service':
    tag => 'pe-workflow-packages',
  }

  @package { 'pe-puppet-enterprise-release':
    tag => [
      'pe-console-packages',
      'pe-master-packages',
      'pe-puppetdb-packages',
      'pe-database-packages',
    ],
  }
}
