# frozen_string_literal: true

require 'net/http'
require 'puppet/ssl'
require 'json'

module PlanRunner
  module Service
    class HttpClient
      attr_reader :client

      DEFAULT_HEADERS = { 'Content-Type': 'application/json' }.freeze

      def initialize(orch_url, config)
        @logger = Logging.logger[self]

        @logger.debug("Create SSL context")
        @ssl_context = Puppet::SSL::SSLProvider.new.create_context(
          cacerts: config[:ca_cert],
          crls: config[:crl],
          private_key: config[:private_key],
          client_cert: config[:cert]
        )

        # Taken from Puppet's Runtime. We do not want to look up
        # Puppet.runtime[:http] because it will share clients between processes.
        @client = Puppet::HTTP::Client.new

        @orch_url = orch_url
      end

      def get_with_orch_cert(url_path, api_token = nil)
        headers = if api_token
                    DEFAULT_HEADERS.merge({ 'X-Authentication': api_token })
                  else
                    DEFAULT_HEADERS
                  end
        uri = handle_orch_uri(url_path)
        handle_request(uri) do |full_url|
          @client.get(
            full_url,
            headers: headers,
            options: { ssl_context: @ssl_context }
          )
        end
      end

      def post_with_orch_cert(url_path, body, api_token = nil)
        headers = if api_token
                    DEFAULT_HEADERS.merge({ 'X-Authentication': api_token })
                  else
                    DEFAULT_HEADERS
                  end
        uri = handle_orch_uri(url_path)
        handle_request(uri, body: body) do |full_url, json_body|
          @client.post(
            full_url,
            json_body,
            headers: headers,
            options: { ssl_context: @ssl_context }
          )
        end
      end

      def put_with_orch_cert(url_path, body, api_token = nil)
        headers = if api_token
                    DEFAULT_HEADERS.merge({ 'X-Authentication': api_token })
                  else
                    DEFAULT_HEADERS
                  end
        uri = handle_orch_uri(url_path)
        handle_request(uri, body: body) do |full_url, json_body|
          @client.put(
            full_url,
            json_body,
            headers: headers,
            options: { ssl_context: @ssl_context }
          )
        end
      end

      # Simliar to get_with_orch_cert, but doesn't assume Orchestrator as the
      # location
      def get_with_cert(url, api_token = nil)
        headers = if api_token
                    DEFAULT_HEADERS.merge({ 'X-Authentication': api_token })
                  else
                    DEFAULT_HEADERS
                  end
        uri = url.is_a?(URI) ? url : URI(url)
        handle_request(uri) do |full_url|
          @client.get(
            full_url,
            headers: headers,
            options: { ssl_context: @ssl_context }
          )
        end
      end

      # This post function is required for puppetdb clients
      def post(url, body:, header:)
        uri = url.is_a?(URI) ? url : URI(url)
        @client.post(uri, body, headers: header, options: { ssl_context: @ssl_context })
      end

      private

      def handle_orch_uri(url_path)
        # Make sure the URL starts with the orch URL, otherwise assume the
        # path was provided without the hostname
        if url_path.include?(@orch_url)
          URI(url_path)
        else
          URI(@orch_url + url_path)
        end
      end

      # Handle URL creation, JSON encoding/decoding, and errors for http requests
      #
      # Expects a fully formed URI object
      def handle_request(uri, body: nil)
        # Don't attempt to JSON.generate a nil body
        json_body = if body.nil?
                      nil
                    else
                      JSON.generate(body)
                    end

        http_response = if body.nil?
                          yield uri
                        else
                          yield uri, json_body
                        end

        # Throw and catch the response error. Throwing allows us to
        # capture if another part of the stack throws this error.
        unless http_response.success?
          raise Puppet::HTTP::ResponseError.new(http_response)
        end

        if http_response.body.nil? || http_response.body.empty?
          ["", http_response.code]
        else
          [JSON.parse(http_response.body), http_response.code]
        end
      rescue Puppet::HTTP::ResponseError => e
        msg = "Orchestrator API request returned HTTP code #{e.response.code}"
        @logger.error(msg)
        # Use trace to print the message in case the request contained something sensitive
        @logger.trace("Failure message: #{e.full_message}")
        [msg, e.response.code]
      rescue StandardError => e
        msg = "Exception #{e.class.name} thrown while attempting API request to Orchestrator"
        @logger.error(msg)
        # Use trace to print the message in case the request contained something sensitive
        @logger.trace("Exception message: #{e.full_message}")
        [msg, 500]
      end
    end
  end
end
