# frozen_string_literal: true

require 'pe_installer/config'
require 'pe_installer/bolt_interface'

module PeInstaller
  module CliExt

    # Common setup code used by CLI commands.
    module Setup
      ENTERPRISE_HIERA_YAML = '/etc/puppetlabs/enterprise/hiera.yaml'

      # Generates the default project directory, a default bolt-project.yaml and an
      # empty inventory.yaml with the necessary role groups for PE, if these
      # artifacts do not yet exist. We use this when initializing bolt.
      def self.prepare_project_dir
        project_dir = PeInstaller::Config.default_project_directory
        FileUtils.mkdir_p(project_dir) if !Dir.exist?(project_dir)

        project_bolt_yaml = "#{project_dir}/bolt-project.yaml"
        if !File.exist?(project_bolt_yaml)
          project_hash = {
            'name'         => 'pe_installer',
            'modulepath'   =>
              '/opt/puppetlabs/installer/share/Boltdir/site-modules:/opt/puppetlabs/installer/share/Boltdir/modules',
            'log'          => {
              "#{PeInstaller::Config.default_log_directory}/#{PeInstaller.tool_name}.log" => {
                'level' => 'debug',
              },
            },
          }
          project_hash['hiera-config'] = ENTERPRISE_HIERA_YAML if File.exist?(ENTERPRISE_HIERA_YAML)
          File.write(project_bolt_yaml, <<~DEFAULT_BOLT_YAML)
            # Default bolt-project.yaml configuration for Puppet Enterprise #{PeInstaller.tool_name} utility.
            #{project_hash.to_yaml}
          DEFAULT_BOLT_YAML
        end

        project_inventory_yaml = "#{project_dir}/inventory.yaml"
        if !File.exist?(project_inventory_yaml)
          File.write(project_inventory_yaml, <<~EMPTY_INVENTORY_YAML)
            # Autogenerated inventory file for Puppet Enterprise #{PeInstaller.tool_name} utility.
            ---
            version: 2
            # NOTE: You may add target information to these groups, but do not modify
            # the group names or remove groups.
            groups:
              - name: infrastructure
                groups:
                  - name: masters
                  - name: databases
                  - name: replicas
                  - name: compilers
          EMPTY_INVENTORY_YAML
        end

        nil
      end

      def self.prepare_logging
        Bolt::Logger.initialize_logging
        bolt_config = PeInstaller::BoltInterface.generate_config
        Bolt::Logger.configure(bolt_config.log, bolt_config.color)
        logger = Logging.logger[self]
        bolt_config.log.each do |log|
          if log.to_a.count > 1 && !log.to_a[1].empty?
            level = log.to_a[1][:level]
            logger.warn("Unknown log level '#{level}'") unless logger.respond_to?(level)
          end
        end

        bolt_config
      end

      # Prepare our environment for execution.
      # @param command [Symbol] the command being invoked.
      # @param thor_options [Hash] the options Hash parsed by Thor from the commandline.
      # @param cmd_args [Hash] additional Hash of arguments provided on the commandline
      #   which were not named flags (so are not in +thor_options+).
      # @return [PeInstaller::Config]
      def setup(command, thor_options, arguments: {})
        PeInstaller::CliExt::Setup.prepare_project_dir
        PeInstaller::CliExt::Setup.prepare_logging

        config = PeInstaller::Config.process(
          command: command,
          cmd_args: arguments,
          thor_options: thor_options,
          global_option_keys: PeInstaller::CLI.global_option_keys
        )
        config.load_externals
        config
      end
    end
  end
end
