#!/opt/puppetlabs/installer/bin/ruby
# This allows us to migrate parameters from one name to another
# after upgrade has completed. Since parameter migration is
# likely a one-time operation, we do it here rather than
# enforcing it on every puppet run. The parameters migrated here
# should have the puppet_enterprise::deprecated_parameter
# suppress_on_install parameter set to true to avoid warnings
# during upgrade.
#
# Currently, this only migrates the
# puppet_enterprise::profile::puppetdb::sync_whitelist parameter
# to sync_allowlist.

$LOAD_PATH << '/opt/puppetlabs/server/data/environments/enterprise/modules/pe_manager/lib'
$LOAD_PATH << '/opt/puppetlabs/server/data/environments/enterprise/modules/pe_install/lib'

require 'puppet'
require 'puppet_x/util/classifier'
require 'puppet_x/util/classification'
require 'puppet_x/util/service_status'
require 'puppet/util/pe_node_groups'

params_to_migrate = [
  {
    :group    => 'PE HA Master',
    :class    => 'puppet_enterprise::profile::puppetdb',
    :oldparam => 'sync_whitelist',
    :newparam => 'sync_allowlist',
  },
  {
    :group    => 'PE Infrastructure Agent',
    :class    => 'puppet_enterprise::profile::agent',
    :oldparam => 'master_uris',
    :newparam => 'primary_uris',
  }
]

# Needed to get SSL context
Puppet.initialize_settings(['--libdir=/dev/null', '--factpath=/dev/null'])

begin
  nc = PuppetX::Util::Classification.get_node_classifier
  all_groups = PuppetX::Util::Classifier.get_groups(nc)
rescue Exception => e
  puts '!!! Error getting data from the classifier !!!'
  puts 'You may encounter warnings about deprecated parameters on your next puppet run.'
  puts 'Please follow the instructions to migrate any deprecated parameters to the new parameter.'
  puts "#{e.full_message}"
end
params_to_migrate.each do |param|
  begin
    group = PuppetX::Util::Classifier.find_group(all_groups, param[:group])
    unless group.nil? || 
            group['classes'].nil? || 
            group['classes'][param[:class]].nil? || 
            group['classes'][param[:class]][param[:oldparam]].nil?
      puts "Migrating #{param[:class]}::#{param[:oldparam]} to #{param[:newparam]}"
      value = group['classes'][param[:class]][param[:oldparam]]
      nc.update_group(
        {
          :id => group['id'],
          :classes => {
            param[:class] => {
              param[:oldparam] => nil,
              param[:newparam] => value,
            }
          }
        }
      )
    end
  rescue Exception => e
    puts "!!! Error migrating parameter #{param[:class]}::#{param[:oldparam]} to #{param[:newparam]} !!!"
    puts "Please migrate your value for #{param[:oldparam]} to #{param[:newparam]} in the #{param[:group]} node group in the classifier."
    puts "You may encounter deprecated parameter warnings on your puppet runs until the parameter is migrated."
    puts "#{e.full_message}"
  end
end
