#!/opt/puppetlabs/installer/bin/ruby
# Verify that secondary infrastructure nodes' pxp-agent is connected directly
# to the primary's PCP broker. If not, then we will run into problems upgrading
# and performing infra actions on secondary nodes.
#
# If the path to an RBAC token is provided, this is loaded and used
# to authenticate with the orchestrator endpoint.
#
# exit codes:
# 0 = All secondary nodes configured correctly (or none present)
# 1 = Unhandled exception
# 4 = RBAC token required
# 5 = No RBAC token at given path
# 6 = RBAC token expired
# 7 = Misconfigured nodes, printed to stdout

$LOAD_PATH << '/opt/puppetlabs/server/data/environments/enterprise/modules/pe_manager/lib'

require 'puppet'
require 'puppet_x/util/service_status'
require 'puppet_x/util/orchestrator'
require 'puppet_x/util/orchestrator_error'
require 'puppet_x/util/rbac'

def load_token(token_path: nil)
  begin
    @rbac_token = PuppetX::Util::RBAC.load_token(token_path)
  rescue RuntimeError => e
    if e.message =~ /Couldn't find an RBAC token/
      exit token_path.nil? ? 4 : 5
    else
      raise
    end
  end
end

# Needed to get SSL context
Puppet.initialize_settings(['--libdir=/dev/null', '--factpath=/dev/null'])

config = PuppetX::Util::ServiceStatus.load_services_config
primary = PuppetX::Util::ServiceStatus.get_master
svcs = config.select { |svc| svc[:type] == 'master' && svc[:node_certname] != primary }
secondary_infra = svcs.map { |svc| svc[:node_certname] }
orch_service = PuppetX::Util::ServiceStatus.get_service_on_primary('orchestrator')

if ARGV.empty?
  @rbac_token = nil
else
  load_token(token_path: File.expand_path(ARGV[0]))
end

begin
  bad_nodes = PuppetX::Util::Orchestrator.get_misconfigured_pcp_broker_connections(orch_service, @rbac_token, secondary_infra, primary)
  bad_nodes = bad_nodes.map { |node| node['name'] }
  if bad_nodes.empty?
    exit 0
  else
    puts bad_nodes.join("\n")
    exit 7
  end
rescue PuppetX::Util::OrchestratorResponseError => e
  if e.message =~ /Route requires authentication/
    load_token
    retry
  elsif e.message =~ /(token has expired|revoked)/
    exit 6
  else
    raise
  end
end
