# frozen_string_literal: true

require 'hocon'
require 'hocon/config_error'
require 'pe_installer/util'

module PeInstaller

  # Errors raised during configuration.
  class ConfigError < PeInstaller::Error; end

  # This class holds commandline inputs parsed from the tool and
  # any other configuration defaults.
  #
  # Bolt configuration is loaded later through {PeInstaller::BoltInterface},
  # with overrides from our config for things like logging, output format,
  # inventory file and so forth.
  class Config
    attr_accessor :command
    attr_accessor :arguments
    attr_accessor :options
    attr_accessor :pe_conf_hash
    attr_accessor :bootstrap_metadata

    # Separates arguments specific to the invoked command from global options.
    #
    # Since both are submitted by the user in the form of '--opt' style options,
    # this method lets us separate the command arguments from general options
    # affecting logging, output, and bolt behavior.
    #
    # @param command [Symbol] the invoked command.
    # @param cmd_args [Hash<String,String>] arguments that were not flags.
    # @param thor_options [Hash<String,Object>] the options hash parsed by
    #   Thor, which will be a mix of command specific and global option flags.
    # @param global_option_keys [Array<String>] list of option keys that are for
    #   global configuration, as opposed to those specifically for the invoked command.
    # @return [PeInstaller::Config]
    def self.process(command:, cmd_args:, thor_options:, global_option_keys:)
      partitioned = thor_options.partition { |k, _v| !global_option_keys.include?(k) }
      arguments, options = partitioned.map(&:to_h)

      overlapped_args = cmd_args.keys & arguments.keys
      if !overlapped_args.empty?
        raise(
          PeInstaller::ConfigError,
          "Overlapping keys (#{overlapped_args}) from arguments: #{cmd_args} and options: #{arguments}"
        )
      end
      arguments.merge!(cmd_args)

      new(
        command: command,
        arguments: arguments,
        options: options
      )
    end

    # Our default bolt project and general workspace.
    def self.default_project_directory
      File.expand_path(File.join('~', '.puppetlabs', PeInstaller.tool_name))
    end

    def self.default_log_directory
      '/var/log/puppetlabs/installer'
    end

    def initialize(command:, arguments: {}, options: {})
      self.command = command
      self.arguments = arguments
      self.options = options
      self.pe_conf_hash = {}
      self.bootstrap_metadata = {}

      _expand_argument_paths

      @logger = Logging.logger[self]
      @loaded = false
    end

    # Loads any external file references passed into the config.
    # @return [PeInstaller::Config] self
    def load_externals
      self.pe_conf_hash = _load_pe_conf
      self.bootstrap_metadata = _load_bootstrap_metadata
      @loaded = true
      self
    end

    # @return [Boolean] true if we've executed {PeInstaller::Config.load_externals}
    def loaded?
      @loaded
    end

    # @return [String] path to pe.conf file if given as a commandline arg.
    def pe_conf_file
      arguments['pe_conf']
    end

    # @api private
    # @return [Hash] The PE tarball's packages/bootstrap-metadata hash if
    # available, otherwise an empty hash.
    # @raise [PeInstaller::Error] for problems loading the file.
    def _load_bootstrap_metadata
      if arguments['pe_tarball_dir'].nil? && arguments['tarball'].nil?
        {}
      else
        PeInstaller::Util.load_bootstrap_metadata_from(
          tarball_dir: arguments['pe_tarball_dir'],
          tarball: arguments['tarball']
        )
      end
    end

    # @api private
    # @return [Hash] the results of parsing {PeInstaller::Config#pe_conf_file}
    # @raise [PeInstaller::ConfigError] if a path to a conf file is set but
    # the file is not a parseable HOCON file.
    def _load_pe_conf
      if pe_conf_file.nil?
        {}
      else
        @logger.warn("The requested --pe_conf file at #{pe_conf_file} does not exist.") if !File.exist?(pe_conf_file)
        Hocon.load(pe_conf_file)
      end
    rescue Hocon::ConfigError => e
      @logger.error(e)
      raise(PeInstaller::ConfigError, "Problem loading or parsing pe.conf file at '#{pe_conf_file}'")
    end

    # Executed during initialization to expand any relative argument paths.
    # @api private
    # @return [Hash] the arguments hash.
    def _expand_argument_paths
      arguments.each_key do |arg|
        # rubocop:disable Style/IfUnlessModifier
        if ['tarball', 'pe_tarball_dir', 'pe_conf'].include?(arg)
          arguments[arg] = File.expand_path(arguments[arg])
        end
        # rubocop:enable Style/IfUnlessModifier
      end
    end
  end
end
