require 'puppet'

class APIHelper
  class BadResponse < RuntimeError; end
  class InvalidType < RuntimeError; end

  attr_accessor :server, :base_path, :port, :options, :headers

  def initialize(port, base_path, server: nil, options: nil, headers: nil)
    # Don't initialize_settings if it's already been done somewhere else first
    Puppet.initialize_settings if Puppet[:confdir].nil?
    @connection = Puppet.runtime[:http]
    @port = port
    @base_path = base_path
    # This will most likely be used on the primary, so default to the local certname
    @server = server || Puppet[:certname]
    @options = options
    @headers = headers
  end

  # NOTE: generally, we don't want to retry puts/posts as they may not be idempotent.
  # The 'body' param is ignored for get, head, and delete.
  # Will use the headers and options set in initialize if not passed in here.
  def request(endpoint, type, body: nil, headers: nil, options: nil, valid_codes: [200], retry_on_invalid_code: false, retry_wait: 5, retry_count: 3)
    raise InvalidType, "Invalid type given: #{type}" if ![:get, :put, :post, :head, :delete].include?(type)
    valid_codes = [valid_codes].flatten
    uri = URI("https://#{@server}:#{@port}/#{@base_path}/#{endpoint}")
    retries = 0
    kwargs = { headers: headers || @headers, options: options || @options }
    valid = false
    while !valid
      if [:put, :post].include?(type)
        response = @connection.send(type, uri, body, **kwargs)
      else
        response = @connection.send(type, uri, **kwargs)
      end

      if valid_codes.include?(response.code)
        valid = true
      else
        if !retry_on_invalid_code || retries > retry_count
          raise BadResponse, "Recieved an invalid response code #{response.code}. Body: #{response.body}"
        end
        retries += 1
        sleep(retry_wait)
      end
    end
    response.body
  end
end
