# Verifies that the node(s) given are of the given role
# $nodes can be a comma-separated array to verify multiple nodes are the same role
# If a node has already been verified for a given role, it is not rechecked.
# When $force = true, skip verification completely. Putting this logic here makes the
# plans a bit cleaner.
function enterprise_tasks::verify_node(TargetSpec $nodes, String $role, Boolean $force) {
  if $force {
    return true
  }
  # If 'localhost' was passed in for $nodes, this will transform it 
  # into the primary's certname.
  $nodes_targets = get_targets($nodes)
  $nodes_munged = $nodes_targets.map |$node| {
    get_target(enterprise_tasks::get_certname($node.host))
  }
  $result = $nodes_munged.reduce(true) |$memo, $node| {
    $roles_verified = $node.vars['roles_verified'] ? {
      undef => [],
      default => $node.vars['roles_verified']
    }
    if $role in $roles_verified {
      $memo and true
    } else {
      $has_role = case $role {
        'primary': {
          enterprise_tasks::node_has_profile($node.host, 'master') and
          enterprise_tasks::node_has_profile($node.host, 'console') and
          !enterprise_tasks::node_has_profile($node.host, 'primary_master_replica')
        }
        'ca': {
          enterprise_tasks::node_has_profile($node.host, 'certificate_authority')
        }
        'puppetdb': {
          enterprise_tasks::node_has_profile($node.host, 'puppetdb') and
          !enterprise_tasks::node_has_profile($node.host, 'master')
        }
        'console': {
          enterprise_tasks::node_has_profile($node.host, 'console') and
          !enterprise_tasks::node_has_profile($node.host, 'master')
        }
        'database': {
          enterprise_tasks::node_has_profile($node.host, 'database') and
          !enterprise_tasks::node_has_profile($node.host, 'master')
        }
        'replica': {
          enterprise_tasks::node_has_profile($node.host, 'primary_master_replica')
        }
        'legacy_compiler': {
          enterprise_tasks::node_has_profile($node.host, 'master') and
          !enterprise_tasks::node_has_profile($node.host, 'console') and
          !enterprise_tasks::node_has_profile($node.host, 'pe_compiler')
        }
        'pe_compiler': {
          enterprise_tasks::node_has_profile($node.host, 'pe_compiler')
        }
        'agent': {
          enterprise_tasks::node_exists_in_pdb($node.host) and
          !enterprise_tasks::node_has_pe_packages($node.host)
        }
        default: { false }
      }
      if $has_role {
        $node.set_var('roles_verified', $roles_verified << $role)
        $memo and true
      } else {
        false
      }
    }
  }
  $nodes_verified = $nodes_munged.filter |$n| { $role in $n.vars['roles_verified'] }
  $nodes_unverified = $nodes_munged.filter |$n| { !($role in $n.vars['roles_verified']) }
  unless $nodes_verified.empty {
    $verified_plural = $nodes_verified.length > 1 ? {
      true => 's were',
      false => ' was',
    }
    enterprise_tasks::message('verify_node', "The following node${verified_plural} verified to have the ${role} role:\n  ${nodes_verified.join("\n  ")}")
  }
  unless $nodes_unverified.empty {
    $unverified_plural = $nodes_unverified.length > 1 ? {
      true => 's have',
      false => ' has',
    }
    enterprise_tasks::message('verify_node', "Could not verify that the following node${unverified_plural} the ${role} role:\n  ${nodes_unverified.join("\n  ")}")
  }
  unless $result {
    fail_plan('Could not verify node roles. If you are sure the given nodes are the correct types, you may bypass this check by passing force=true to the plan. If this plan was run from a puppet infrastructure command, you may use the --force flag.')
  }
}
