#! /usr/bin/env bash

set -e

# shellcheck disable=SC2154
platform_tag=$PT_platform_tag
# shellcheck disable=SC2154
pe_version=$PT_version
# shellcheck disable=SC2154
pe_family=$PT_family
# shellcheck disable=SC2154
workdir=${PT_workdir:-/root}

if [ -z "$pe_version" ] && [ -z "$pe_family" ]; then
  echo "Must set either version or family" >&2
  exit 1
fi

cd "$workdir"

if [ -n "$pe_version" ]; then
  pe_family=$(echo "$pe_version" | grep -oE '^[0-9]+\.[0-9]+')
fi

if [[ "$pe_version" =~ ^[0-9]+\.[0-9]+\.[0-9]+$ ]]; then
  release_version='true'
  base_url="https://artifactory.delivery.puppetlabs.net/artifactory/generic_enterprise__local/archives/releases/${pe_version}"
elif [[ "$pe_version" =~ ^[0-9]+\.[0-9]+\.[0-9]+-rc[0-9]+$ ]]; then
  release_version='true'
  base_url="https://artifactory.delivery.puppetlabs.net/artifactory/generic_enterprise__local/archives/internal/${pe_family}"
else
  base_url="https://artifactory.delivery.puppetlabs.net/artifactory/generic_enterprise__local/${pe_family}/ci-ready"
  if ! curl --fail --silent "${base_url}/LATEST" > /dev/null;  then
    # Most likely this PE family is actually the main branch
    base_url="https://artifactory.delivery.puppetlabs.net/artifactory/generic_enterprise__local/main/ci-ready"
  fi
fi

if [ -z "$pe_version" ]; then
  pe_version=$(curl --fail --silent "${base_url}/LATEST")
  version_lookup_code=$?
  if [ ${version_lookup_code} != 0 ]; then
    error="No LATEST file found under ${base_url}"
  else
    pe_family_check=$(echo "$pe_version" | grep -oE '^[0-9]+\.[0-9]+')
    if [ "${pe_family}" != "${pe_family_check}" ]; then
      error="Specified PE family '${pe_family}' did not match PE family determined from '${base_url}/LATEST': ${pe_family_check}."
    fi
  fi
  if [ -n "${error}" ]; then
    echo "{
  \"_error\": {
    \"msg\": \"${error}\",
    \"kind\": \"enterprise_tasks/get_pe\",
    \"details\": {
      \"curl_exit_code\": \"${version_lookup_code}\",
      \"pe_family\": \"${pe_family}\",
      \"pe_version\": \"${pe_version}\",
      \"base_url\": \"${base_url}\"
    }
  }
}"
    exit 1
  fi
fi

pe_dir="puppet-enterprise-${pe_version}-${platform_tag}"

if [ "$release_version" == 'true' ]; then
  pe_tarball="${pe_dir}.tar.gz"
else
  pe_tarball="${pe_dir}.tar"
fi

pe_tarball_url="${base_url}/${pe_tarball}"

curl_code=0
tar_code=0

set +e
if [ ! -f "${pe_tarball}" ]; then
  curl --silent --output "${pe_tarball}" "${pe_tarball_url}"
  curl_code=$?
fi
if [ ! -d "${pe_dir}" ]; then
  tar -xf "${pe_tarball}"
  tar_code=$?
fi
set -e

if [ "$curl_code" != 0 ] || [ "$tar_code" != 0 ]; then
  echo "{
  \"_error\": {
    \"msg\": \"Failed either to curl or untar the PE tarball from ${pe_tarball_url}\",
    \"kind\": \"enterprise_tasks/get_pe\",
    \"details\": {
      \"curl_exit_code\": \"${curl_code}\",
      \"tar_exit_code\": \"${tar_code}\",
      \"pe_tarball_url\": \"${pe_tarball_url}\",
      \"pe_tarball\": \"${pe_tarball}\",
      \"pe_dir\": \"${pe_dir}\"
    }
  }
}"
  exit 1
fi

echo "{
  \"workdir\":\"${workdir}\",
  \"pe_dir\":\"${workdir}/${pe_dir}\",
  \"pe_tarball\":\"${pe_tarball}\",
  \"pe_tarball_url\":\"${pe_tarball_url}\",
  \"pe_family\":\"${pe_family}\",
  \"pe_version\":\"${pe_version}\"
}"
