#!/opt/puppetlabs/puppet/bin/ruby

require_relative '../files/enterprise_task_helper.rb'
require 'json'
require 'open3'
require 'etc'

class ProvisionReplica < EnterpriseTaskHelper

  def task(host:, replication_timeout: 1800, streaming: true, enable: false, topology: nil, skip_agent_config: false, agent_server_urls: nil, pcp_brokers: nil, token_file: nil, **_kwargs)
    puppet_infra = '/opt/puppetlabs/bin/puppet-infrastructure'
    total_output = ''

    command = [
      puppet_infra,
      'provision',
      'replica',
      host,
    ]
    command += ['--token-file', token_file] if !token_file.nil?

    streaming_flag = streaming ? '--streaming' : '--no-streaming'
    command += [streaming_flag]

    if enable
      raise EnterpriseTaskHelper::Error.new("Must provide the 'topology' parameter when 'enable' is true.", 'puppetlabs.provision-replica/invalid-parameters') unless topology
      command += ['--enable', '--yes', '--topology', topology]
      if topology == 'mono-with-compile'
        raise EnterpriseTaskHelper::Error.new('Must provide either skip_agent_config=true or skip_agent_config=false and both agent_server_urls and pcp_brokers.', 'puppetlabs.provision-replica/invalid-parameters') unless skip_agent_config || (agent_server_urls && pcp_brokers)
        if skip_agent_config
          command << '--skip-agent-config'
        else
          command << '--agent-server-urls' << agent_server_urls
          command << '--pcp-brokers' << pcp_brokers
        end
      end
    end

    output, status = Open3.capture2e(*command)
    total_output += output
    raise EnterpriseTaskHelper::Error.new("Failed to provision replica with certname #{host}", 'puppetlabs.provision-replica/provision-failed', 'output' => output) if !status.success?
    # The --enable flag does this checking already when provisioning
    unless enable
      start_time = Time.now
      services_ok = false

      while (Time.now - start_time) < replication_timeout && !services_ok
        command = [
          puppet_infra,
          'status',
          '--host',
          host,
        ]
        output, _stderr, status = Open3.capture3(*command)
        total_output += "\n" + output
        break if !status.success?

        services_up, services_total = output.scan(%r{([0-9]+) of ([0-9]+) services are fully operational}).flatten
        break if services_up.nil? || services_total.nil?

        if services_up == services_total && services_up != 0
          services_ok = true
        else
          sleep(15)
        end
      end

      if !services_ok
        raise EnterpriseTaskHelper::Error.new(
          "Replica status check timed out or failed on host with certname #{host}",
          'puppetlabs.provision-replica/provision-timedout',
          {
            puppet_infra_status_output: output.split("\n"),
            exitcode: status.exitstatus,
            services_up: services_up,
            services_total: services_total,
          },
        )
      end
    end

    result = {
      success: true,
      _output: total_output,
    }
    result.to_json
  end
end

# The HOME variable is required to run the provision command but may not exist
# when this task is executed using the orchestrator
ENV['HOME'] ||= Etc.getpwuid.dir # rubocop:disable Style/EnvHome (rule does not take into account assignment)

ProvisionReplica.run if __FILE__ == $PROGRAM_NAME
