#!/opt/puppetlabs/puppet/bin/ruby
# frozen_string_literal: true

require_relative '../files/enterprise_task_helper.rb'
require 'json'
require 'open3'

class PuppetserverCASign < EnterpriseTaskHelper

  # Provides information about a particular certificate, given the
  # output of `puppetserver cert list --certname <certname>`
  class CertStatus
    attr_accessor :certname, :alt_names, :signed, :requested

    def initialize(ca_output)
      header, certline = ca_output.split("\n")
      case header
      when 'Signed Certificates:'
        self.signed = true
      when 'Requested Certificates:'
        self.requested = true
      end
      contents = certline.nil? ?
        [] :
        certline.strip.scan(/^[^\s]+|(?<=DNS:)[^"]+/)
      self.certname = contents.first
      self.alt_names = contents[1..-1] || []
    end

    # Eliminate nils
    def signed?; !!signed end
    def requested?; !!requested end

    def found?
      signed? || requested?
    end
  end

  def task(certname:, allow_subject_alt_names: false, signing_timeout: 2, dns_alt_names: nil, **_kwargs)
    # Right now, puppetserver ca does not support passing a flag for allowing
    # SAN or auth extensions for a one-off cert signing. We can detect if a CSR
    # contains alt names in the request by looking at the output of
    # puppetserver ca list. However, auth extensions are not listed here.  So
    # currently, allow_authorized_extensions is an unused argument.
    #
    # Note that if autosigning is enabled, this task will not be able to catch
    # a CSR that contains alt names when allow_subject_alt_names == false

    certnames = [certname].flatten
    puppetserver_bin = '/opt/puppetlabs/server/bin/puppetserver'

    certname_cmd = ['/opt/puppetlabs/bin/puppet', 'config', 'print', 'certname']
    caserver = Open3.capture2e(*certname_cmd)[0].strip

    errors = []
    successful_nodes = []
    certnames.each do |c|
      signed = false
      cert = CertStatus.new('')
      command = []
      output = nil
      status = nil
      list_cmd = [puppetserver_bin, 'ca', 'list', '--certname', c]
      5.times do
        output, _stderr, status = Open3.capture3(*list_cmd)
        cert = CertStatus.new(output)
        break if cert.found?
        sleep signing_timeout
        signing_timeout *= 2
      end

      if !cert.found?
        errors << [
          "No CSR detected for certname #{c}",
          {
            command: command.join("\n"),
            output: output,
            exitcode: status.exitstatus,
          },
        ]
        next
      end

      if cert.requested? && !cert.alt_names.empty? && !allow_subject_alt_names
        errors << [
          "CSR for #{c} contained the following subject alt names and allow_subject_alt_names == false: #{cert.alt_names.join(', ')}",
        ]
        next
      end

      command = [puppetserver_bin, 'ca', 'sign', '--certname', c]
      output, _stderr, status = Open3.capture3(*command) if cert.requested?
      signed = true if cert.signed? || status&.success?
      if !signed
        errors << [
          "Could not sign request for certname #{c} using caserver #{caserver}",
          'output' => output,
        ]
        next
      end

      if !dns_alt_names.nil? && !dns_alt_names.empty?
        list_output, _list_stderr, _list_status = Open3.capture3(*list_cmd)
        errored = false
        dns_alt_names.split(',').each do |name|
          next if list_output.include?("DNS:#{name}")
          errors << [
            "Cert for #{c} does not contain the dns_alt_names '#{name}'",
            'output' => list_output,
          ]
          errored = true
        end
        next if errored
      end

      successful_nodes << c
    end

    if errors.empty?
      { signed: successful_nodes }.to_json
    else
      raise EnterpriseTaskHelper::Error.new(
        "One or more errors occurred attempting to sign the given certname(s): #{errors.join("\n")}",
        'puppetlabs.puppetserver-ca-sign-errors',
        'errored_certnames' => certnames - successful_nodes,
        'errors' => errors,
      )
    end
  end
end

PuppetserverCASign.run if __FILE__ == $PROGRAM_NAME
