# This plan runs over the orchestrator, and is used by the puppet infrastructure
# upgrade action to upgrade compilers and replicas. If --all is specified,
# PuppetDB queries will be performed to find all legacy and PE Compilers. If a
# node is passed in, it checks if that node is a compiler first using the same
# queries. For legacy compilers, this is finding nodes with the 'master' profile
# applied, and subtracting out the primary and replica. For PE Compilers, this is
# looking for nodes with the pp_auth_role = pe_compiler trusted fact.
# Steps the plan will take:
# 1. pe_build is value in the pe_build fact on the primary, which is read from
#    /opt/puppetlabs/server/pe_build
# 2. If the agent currently on the node is less than version 6.6.0, we must
#    upgrade the agent first in order to support multi-file tasks that the
#    upgrade_secondary plan relies on.  If you are doing this process manually,
#    you don’t need to worry about this.  On the target:
#      a. Disable the agent with puppet agent --disable
#      b. Run curl --cacert /etc/puppetlabs/puppet/ssl/certs/ca.pem
#         https://<primary>:8140/packages/<pe_build>/install.bash | sudo bash
#      c. Run puppet facts upload to update facts for the node so the
#         orchestrator knows we have a new pxp-agent.
#      d. The agent is purposefully left disabled at this point, as we don’t
#         want a background run to happen yet.
# 3. Disable the agent on the target with puppet agent --disable.
# 4. Stop all PE services on the target.
# 5. If not done previously, run curl --cacert /etc/puppetlabs/puppet/ssl/certs/ca.pem
#    https://<primary>:8140/packages/<pe_build>/install.bash | sudo bash
# 6. Restart pxp-agent on the target.
# 7. Run FACTER_pe_build=<pe_build> puppet agent -t --tags
#    puppet_enterprise_repo_update on the target.
# 8. Run FACTER_pe_build=<pe_build> puppet agent -t on the target.
# 9. Re-enable the agent on the target with puppet agent --enable.
#
# @api private
plan enterprise_tasks::upgrade_secondary(
  TargetSpec $target,
  Optional[TargetSpec] $primary = 'localhost',
  Optional[Boolean] $force      = false,
) {
  $constants = constants()
  enterprise_tasks::test_connection([$target, $primary])
  $status_hash = run_plan(enterprise_tasks::get_service_status, target => $target,
    service    => 'puppet',
  )
  $result_or_error = catch_errors() || {
    enterprise_tasks::with_agent_disabled([$target]) || {
      $primary_certname = run_command("${constants['puppet_bin']} config print certname", $primary).first['stdout'].strip
      enterprise_tasks::verify_node($primary, 'primary', $force)

      # The puppet infra upgrade command for which this plan is intended to be used with is already
      # verifying that the node given is the correct type, so we don't need to recheck here.

      $primary_server_build = run_command('cat /opt/puppetlabs/server/pe_build', $primary).first['stdout'].strip

      [$target].flatten.each |$t | {
        run_task(enterprise_tasks::pe_services, $t,
          role  => enterprise_tasks::get_node_role($t),
          state => 'stopped',
        )
      }

      # PE-33668/PE-33669 There is a bug in orch that may cause this run_command to error due to
      # pxp-agent being down, even though the command finishes fine. Detect this and move on.
      $install_bash_result = run_command("/opt/puppetlabs/puppet/bin/curl --cacert ${constants['ca_pem']} https://${primary_certname}:8140/packages/current/install.bash | bash", $target,
        '_catch_errors' => true
      ).first
      if !$install_bash_result.ok and $install_bash_result.error.msg !~ /Not connected/ {
        enterprise_tasks::message('upgrade_secondary', '!!! Error running install.bash')
        fail_plan("${install_bash_result}") # lint:ignore:only_variable_string We're coercing a string out of the result object here. It's possible this should be a String() or we should pull something specific out of the result instead.
      }
      # Because install.bash should restart pxp-agent, wait for it to come back here. Also,
      # if we hit the bug in PE-33668, this waits until install.bash actually finishes
      # and pxp-agent is restarted.
      wait_until_available($target)
      # Probably don't really need to do this, but just in case install.bash didn't correctly detect
      # we're doing an upgrade and didn't restart pxp-agent.
      run_task(service, $target, action => 'restart', name => 'pxp-agent')
      wait_until_available($target)

      # Update the puppet_enterprise repo before applying any other updates
      run_task(enterprise_tasks::run_puppet, $target,
        env_vars => { 'FACTER_pe_build' => $primary_server_build },
        additional_args => ['--tags','puppet_enterprise_repo_update'],
      )
      run_task(enterprise_tasks::run_puppet, $target,
        env_vars => { 'FACTER_pe_build' => $primary_server_build },
      )
      $primary_server_pe_modules_version = run_task(enterprise_tasks::get_package_version, $primary, package => 'pe-modules').first().value()['version']
      $secondary_pe_modules_version = run_task(enterprise_tasks::get_package_version, $target, package => 'pe-modules').first().value()['version']
      if $primary_server_pe_modules_version != $secondary_pe_modules_version {
        enterprise_tasks::message('upgrade_secondary',"ERROR: The target did not have the expected pe-modules version, had ${secondary_pe_modules_version} expected to match the primary server's version of ${primary_server_pe_modules_version}" )
        fail_plan("Failed on ${target[0]}", 'target did not have an upgraded pe-modules package')
      }
    }
  }
  enterprise_tasks::message('upgrade_secondary','Applying original puppet service state...')
  run_command("${constants['puppet_bin']} resource service puppet ensure=${status_hash[status]} enable=${status_hash[enabled]}", $target)
  if $result_or_error =~ Error {
    return fail_plan($result_or_error)
  }
}
