#!/opt/puppetlabs/puppet/bin/ruby
# frozen_string_literal: true

require 'json'
require 'open3'
require 'fileutils'

def sql_2019_3
  <<~EOF
    BEGIN TRANSACTION;

    ALTER TABLE certnames DROP CONSTRAINT IF EXISTS certnames_reports_id_fkey;
    UPDATE certnames SET latest_report_id = NULL;
    TRUNCATE TABLE reports CASCADE;

    ALTER TABLE certnames
      ADD CONSTRAINT certnames_reports_id_fkey
         FOREIGN KEY (latest_report_id) REFERENCES reports(id) ON DELETE SET NULL;

    COMMIT TRANSACTION;
  EOF
end

def sql_2019_6
  <<~EOF
    BEGIN TRANSACTION;

    ALTER TABLE certnames DROP CONSTRAINT IF EXISTS certnames_reports_id_fkey;
    UPDATE certnames SET latest_report_id = NULL;

    DO $$ DECLARE
        r RECORD;
    BEGIN
        FOR r IN (SELECT tablename FROM pg_tables WHERE tablename LIKE 'resource_events_%') LOOP
            EXECUTE 'DROP TABLE ' || quote_ident(r.tablename);
        END LOOP;
    END $$;

    TRUNCATE TABLE reports CASCADE;

    ALTER TABLE certnames
      ADD CONSTRAINT certnames_reports_id_fkey
        FOREIGN KEY (latest_report_id) REFERENCES reports(id) ON DELETE SET NULL;

    COMMIT TRANSACTION;
  EOF
end

class DeleteReports
  attr_accessor :pe_version, :sql_file

  # This task needs to run on agents < 6.6.0 which does not allow multiple files in tasks
  class Error < RuntimeError
    attr_reader :kind, :details, :issue_code

    def initialize(msg, kind, details = nil)
      super(msg)
      @kind = kind
      @issue_code = issue_code
      @details = details || {}
    end

    def to_h
      { 'kind' => kind,
        'msg' => message,
        'details' => details }
    end
  end

  def self.run
    input = $stdin.read
    params = JSON.parse(input)
    params = params.each_with_object({}) { |e, hash| hash[e[0].to_sym] = e[1] }
    result = new.task(**params)
    $stdout.print JSON.generate(result)
  rescue DeleteReports::Error => e
    $stdout.print({ _error: e.to_h }.to_json)
    exit 1
  rescue StandardError => e
    error = DeleteReports::Error.new(e.message, e.class.to_s, 'stacktrace' => e.backtrace)
    $stdout.print({ _error: error.to_h }.to_json)
    exit 1
  end

  def initialize
    version_file = '/opt/puppetlabs/server/pe_version'
    @pe_version = File.read(version_file).strip.to_s
    @sql_file = '/tmp/delete-reports.sql'
  end

  def create_sql_file(sql)
    file = File.open(sql_file, 'w')
    file.puts(sql)
    FileUtils.chown 'pe-postgres', 'pe-postgres', sql_file
    file.close
  end

  def get_cmd
    version = Gem::Version.new(pe_version)
    cmd = %(su - pe-postgres -s /bin/bash -c "/opt/puppetlabs/server/bin/psql -d pe-puppetdb -f #{sql_file}")
    if version >= Gem::Version.new('2019.7') || (version >= Gem::Version.new('2018.1.15') && version < Gem::Version.new('2019.0'))
      cmd = ['/opt/puppetlabs/bin/puppetdb', 'delete-reports']
    elsif version >= Gem::Version.new('2019.3') && version < Gem::Version.new('2019.7')
      create_sql_file(sql_2019_6)
    else
      create_sql_file(sql_2019_3)
    end
    cmd
  end

  def stop_service(service)
    stop_service = ['/opt/puppetlabs/bin/puppet', 'resource', 'service', service, 'ensure=stopped']
    stdout, _stderr, _status = Open3.capture3(*stop_service)
    raise DeleteReports::Error.new("Stopping service #{service} failed", 'puppetlabs.delete-reports/stop-service-failed') if !stdout.strip.include?('stopped')
  end

  def task(_)
    output = ''
    _, postgres_status = Open3.capture2e("/opt/puppetlabs/bin/puppet resource package | grep \"package { 'pe-postgres\"")
    if postgres_status.exitstatus.zero?
      services = ['pe-puppetdb', 'pe-puppetserver', 'puppet']
      services.each do |service|
        stop_service(service)
      end
      cmd = get_cmd
      output, status = Open3.capture2e(*cmd)
      FileUtils.rm_f(sql_file)
      raise DeleteReports::Error.new("Deleting reports from puppetdb failed with error: #{output}", 'puppetlabs.delete-reports/delete-reports-failed', 'output' => output) if !status.exitstatus.zero?
    else
      output = 'No postgres packages detected. Skipping report deletion'
    end
    result = { _output: output }
    return result.to_json
  end
end
DeleteReports.run if __FILE__ == $PROGRAM_NAME
