require 'puppet'
class PEPostgresqlInfo

  # Constants get redefined after a Facter.reset.
  # Having this available for mocking aids testing.
  def self.pe_server_dir; '/opt/puppetlabs/server'; end
  def pe_server_dir; PEPostgresqlInfo.pe_server_dir; end
  def pg_data_dir; "#{pe_server_dir}/data/postgresql"; end
  def pg_app_dir; "#{pe_server_dir}/apps/postgresql"; end

  def databases
    %w{
      activity
      classifier
      orchestrator
      puppetdb
      rbac
      inventory
    }.freeze
  end

  def packages
    unless @packages
      @packages = {}
      os_package_providers = [:apt, :yum, :dnf, :zypper] # :dpkg, :rpm
      Puppet::Type.type(:package).providers_by_source.each do |provider|
        next unless os_package_providers.include? provider.name
        begin
          provider.instances.each do |resource_instance|
            properties = resource_instance.properties
            if properties[:name].start_with?('pe-postgresql')
              @packages[properties[:name]] = properties[:ensure]
            end
          end
        rescue => detail
          Puppet.log_exception(detail, "Cannot collect packages for #{provider} provider; #{detail}")
        end
      end
    end

    @packages
  end

  # This hash will return only the pe-postgresql* packages (pe-postgresql,
  # pe-postgresql-server, etc.). It modifies the full package version to strip
  # the PE version and platform information.
  #
  # So, for example:
  #
  #   2019.0.9.6.10-2.pe.el7
  #
  # will be returned with just the Postgresql version and the release:
  #
  #   9.6.10.2
  #
  # @return [Hash] a hash of installed pe-postgresql* packages and
  #   their current Postgres versions.
  def installed_packages
    packages.each_with_object({}) do |pair,hash|
      package, ver = pair
      version_regex = /\A(?:\d{4}\.\d+\.)?(\d+\.\d+(?:\.\d+)?-\d+)/
      if matcher = version_regex.match(ver)
        hash[package] = matcher[1]
      end
    end
  end

  # PE Postgresql versions were of the form 9.4.7-1 prior to 2016.5, at which
  # point they were of the form 2016.5.9.4.9-1. The first two
  # numbers are the PE release numbers. In addition, there is trailing platform
  # specific text in the version string returned from the +ensure+ property.
  #
  # All we return is the major.minor of the postgresql version (9.4 in the
  # above examples)
  def installed_server_version
    latest_server_package_version = installed_packages.select do |name,v|
      name =~ /\Ape-postgresql(?:\d{2})?-server\Z/
    end.values.sort do |a,b|
      Gem::Version.new(a) <=> Gem::Version.new(b)
    end.last
    if !latest_server_package_version.nil?
      version_regex = /\A((?:(?:9)\.\d+)|1\d)\./
      if matcher = version_regex.match(latest_server_package_version)
        matcher[1]
      end
    end
  end

  # @return [Array<String>] All of the Postgres version data dirs present in
  #   /opt/puppetlabs/server/data/postgresql, sorted in version order, oldest
  #   to newest.
  def versions
    unless @versions
      @versions = []
      if Dir.exist?(pg_data_dir)
        entries = Dir.entries(pg_data_dir)
        @versions = entries.select do |e|
          e.match(/\A\d+(?:\.\d+)?\Z/)
        end
        @versions.sort! do |a,b|
          Gem::Version.new(a) <=> Gem::Version.new(b)
        end
      end
    end
    @versions
  end

  # @return [String] to the base Postgresql apps directory if version
  #   is the installed version, otherwise the versioned app directory
  #   of a backup, if it exists on disk.
  def app_dir(version)
    @app_dirs ||={}
    if version && !@app_dirs.include?(version)
      versioned_appdir = "#{pg_app_dir}/#{version}"

      if Dir.exist?(versioned_appdir)
        @app_dirs[version] = versioned_appdir
      elsif version == installed_server_version
        @app_dirs[version] = pg_app_dir
      end
    end
    @app_dirs[version]
  end

  def data_dir(version)
    @data_dirs ||= {}
    if version && !@data_dirs.include?(version)
      versioned_datadir = "#{pg_data_dir}/#{version}"
      if Dir.exist?(versioned_datadir)
        @data_dirs[version] = versioned_datadir
      end
    end
    @data_dirs[version]
  end

  def tablespace_dirs(version)
    @tablespace_dirs ||= {}
    if !@tablespace_dirs.include?(version)
      @tablespace_dirs[version] = databases.map do |db|
        tablespace_name = tablespace(version)
        versioned_tablespace_dir = "#{pg_data_dir}/#{db}/#{tablespace_name}"
        (!tablespace_name.nil? && Dir.exist?(versioned_tablespace_dir)) ?
          versioned_tablespace_dir :
          nil
      end.compact
      @tablespace_dirs[version].freeze
    end
    @tablespace_dirs[version]
  end

  def files(version)
    file_map = {}
    file_map['app_dir'] = app_dir(version)
    file_map['data_dir'] = data_dir(version)
    file_map['tablespaces'] = tablespace_dirs(version)
    file_map
  end

  # @param directory [String] a directory on the system.
  # @return [Integer] count of bytes used by all files within
  #   that directory. Excludes symlinks. Returns nil if unable to read
  def get_bytes_used(directory)
    if ! (directory && Dir.exist?(directory))
      return 0
    end
    du_output = Puppet::Util::Execution.execute("du --block-size=1024 -s #{directory}", :failonfail => false, :combine => true)
    exit_code = du_output.exitstatus
    if exit_code != 0
      Facter.warn("Failed disk used check in #{directory} (exit:#{exit_code}):\n#{du_output}")
    end
    # $ du --block-size=1024 -s /opt/puppetlabs/server/data/postgresql/9.6/data
    # 63460   /opt/puppetlabs/server/data/postgresql/9.6/data
    # du can have errors when temporary files are removed during scanning.
    # This splits the output up and extracts the bytes it was able to find
    block_count = encode_utf8(du_output).split("\n")[-1].split("\t")[0].to_i
    block_count * 1024
  end

  # @param version [String] The major.minor Postgres version number
  # @return [Integer] bytes used by the data and tablespace directories
  #   of a given Postgresql version instance (current or backup). Or 0
  #   if directories not found.
  def used_bytes(version)
    file_map = files(version)
    directories = file_map['tablespaces'].dup
    directories << file_map['data_dir']
    directories.compact!

    directories.inject(0) do |sum,d|
      sum += get_bytes_used(d)
      sum
    end
  end

  # @return [Hash] of the app, data and tablespace directories and used bytes
  #   for a given postgres version.
  def version_data(version)
    version_data = {}
    version_data[version] = files(version).merge({
      'used_bytes' => used_bytes(version),
    })
  end

  # There is one tablespace directory in the #pg_data_dir for each of the #databases.
  # However the actual tablespace directory is a versioned subdirectory within these,
  # versioned by Postgres version: PG_9.4_201608131, the major.minor and _serverversion.
  #
  # @param version [String] The major.minor Postgres version number ('9.4' for example)
  # @return [String] The versioned tablespace directory name for the given postgres version.
  def tablespace(version)
    @tablespace_names ||= {}
    unless @tablespace_names.include?(version)
      activity_tablespace_dir = "#{pg_data_dir}/activity"
      if Dir.exist?(activity_tablespace_dir)
        path = Dir.glob("#{activity_tablespace_dir}/PG_#{version}_*").first
        name = File.basename(path) if path
      end
      @tablespace_names[version] = name
    end
    @tablespace_names[version]
  end

  # @return [Hash] of the :size_bytes, :available_bytes, :used_bytes for the
  #   partition holding the postgresql data.
  def data_partition_size
    dps = {
      :size_bytes => 0,
      :used_bytes => 0,
      :available_bytes => 0,
    }
    df_output = Puppet::Util::Execution.execute("df -P --block-size=1024 #{pg_data_dir}", :failonfail => false, :combine => true)
    exit_code = df_output.exitstatus
    if exit_code != 0
      Facter.debug("Failed disk free check in #{pg_data_dir} (exit: #{exit_code}):\n#{df_output}")
      return dps
    end

    # $ df -P --block-size=1024 /opt/puppetlabs/server/data/postgresql/
    # Filesystem                   1024-blocks    Used Available Capacity Mounted on
    # /dev/mapper/VolGroup-lv_root    19003260 3677388  14353892      21% /
    last_line = encode_utf8(df_output).split("\n").last
    values = last_line.split(/\s+/).reject { |v| v.to_i == 0 }

    dps[:size_bytes] = values[0].to_i * 1024
    dps[:used_bytes] = values[1].to_i * 1024
    dps[:available_bytes] = values[2].to_i * 1024

    dps
  end

  def encode_utf8(string)
    string.encode('UTF-8', :invalid => :replace)
  end

  def to_hash
    dps = data_partition_size

    versions_hash = versions.inject({}) do |hash,version|
      hash[version] = version_data(version)
      hash
    end

    {
      'installed_server_version' => installed_server_version,
      'installed_packages' => installed_packages,
      'data_partition_size_bytes' => dps[:size_bytes],
      'data_partition_available_bytes' => dps[:available_bytes],
      'versions' => versions_hash,
    }
  end
end
