# Manage a config file that contains information about all known services in a
# PE installation; this is very useful for CLI tools that have to talk to
# various pieces of it.
#
# @param path [String] The path to the config file
# @param user [String] The user that owns the config file
# @param group [String] The group that owns the config file
# @param mode [String] The mode of the config file
# @param additional_services Service records to add to the config file,
# in addition to those queried from puppetdb.
# @param additional_nodes Node records to add to the config file, in addition
# to those queried from pe.conf
class puppet_enterprise::cli_config (
  String $path,
  String $user,
  String $group,
  String $mode,
  Array[
    Struct[
      { type => String,
        url => String,
        server => String,
        port => Integer,
        prefix => String,
        status_url => String,
        status_prefix => String,
        primary => Boolean,
        node_certname => String }]] $additional_services = [],
  Array[
    Struct[
      { certname => String,
        role => String }]] $additional_nodes = [],
) inherits puppet_enterprise {
  if $settings::storeconfigs {
    $classes_to_fetch = {
      'Puppet_enterprise::Profile::Master' => [
        { type => 'master',
          port_param => 'ssl_listen_port',
          service_prefix => '',
          status_key => 'pe-master',
          display_name => 'Puppet Server',},],
      'Puppet_enterprise::Master::File_sync' => [
        { type => 'file-sync-storage',
          port_param => 'puppetserver_webserver_ssl_port',
          service_prefix => '',
          status_key => 'file-sync-storage-service',
          disabled_param => 'storage_service_disabled',
          display_name => 'File Sync Storage Service',},
        { type => 'file-sync-client',
          port_param => 'puppetserver_webserver_ssl_port',
          service_prefix => '',
          status_key => 'file-sync-client-service',
          display_name => 'File Sync Client Service',},],
      'Puppet_enterprise::Master::Code_manager' => [
        { type => 'code-manager',
          port_param => 'webserver_ssl_port',
          service_prefix => '',
          status_key => 'code-manager-service',
          status_port_param => 'puppet_master_port',
          display_name => 'Code Manager',}],
      'Puppet_enterprise::Profile::Puppetdb' => [
        { type => 'puppetdb',
          port_param => 'ssl_listen_port',
          service_prefix => 'pdb',
          status_key => 'puppetdb-status',
          display_name => 'PuppetDB'}],
      'Puppet_enterprise::Profile::Orchestrator' => [
        { type => 'orchestrator',
          port_param => 'ssl_listen_port',
          service_prefix => 'orchestrator',
          status_key => 'orchestrator-service',
          display_name => 'Orchestrator'},
        { type => 'pcp-broker',
          port_param => 'pcp_listen_port',
          status_port_param => 'ssl_listen_port',
          service_prefix => 'pcp',
          status_key => 'broker-service',
          display_name => 'PCP Broker',
          protocol => 'wss',
          status_protocol => 'https'},
        { type => 'pcp-broker',
          port_param => 'pcp_listen_port',
          status_port_param => 'ssl_listen_port',
          service_prefix => 'pcp2',
          status_key => 'broker-service',
          display_name => 'PCP Broker v2',
          protocol => 'wss',
          status_protocol => 'https'}],
      'Puppet_enterprise::Profile::Console' => [
        { type => 'classifier',
          port_param => 'console_services_api_ssl_listen_port',
          service_prefix => 'classifier-api',
          status_key => 'classifier-service',
          display_name => 'Classifier',},
        { type => 'rbac',
          port_param => 'console_services_api_ssl_listen_port',
          service_prefix => 'rbac-api',
          status_key => 'rbac-service',
          display_name => 'RBAC',},
        { type => 'activity',
          port_param => 'console_services_api_ssl_listen_port',
          service_prefix => 'activity-api',
          status_key => 'activity-service',
          display_name => 'Activity Service',}
      ],
      'Puppet_enterprise::Master::Host_action_collector' => [
        { type => 'pe-host-action-collector',
          port_param => 'ssl_listen_port',
          service_prefix => '',
          status_key => 'pe-host-action-collector',
          display_name => 'PE Host Action Collector Service',
          disabled_param => 'disable_service',},
      ],
      'Puppet_enterprise::Profile::Patching_service' => [
        { type => 'pe-patching-service',
          port_param => 'ssl_listen_port',
          service_prefix => '',
          status_key => 'pe-patching-service',
          display_name => 'PE Patching Service',
          disabled_param => 'disable_service',},
      ],
      'Puppet_enterprise::Profile::Infra_assistant' => [
        { type => 'pe-infra-assistant',
          port_param => 'ssl_listen_port',
          service_prefix => '',
          status_key => 'pe-infra-assistant',
          display_name => 'PE Infra Assistant',
          disabled_param => 'disable_service',},
      ],
      'Puppet_enterprise::Profile::Workflow_service' => [
        { type => 'pe-workflow-service',
          port_param => 'ssl_listen_port',
          service_prefix => '',
          status_key => 'pe-workflow-service',
          display_name => 'PE Workflow Service',
          disabled_param => 'disable_service',},
      ],
      # PostgreSQL doesn't have a status endpoint, but this lets us
      # track a managed database node in the services.conf.
      'Puppet_enterprise::Profile::Database' => [
        { type => 'postgresql',
          # Port of a managed pe-postgresql node is not configureable.
          port_default => 5432,
          service_prefix => '',
          status_prefix => '',
          status_key => 'pe-postgresql',
          display_name => 'PostgreSQL',
          protocol => 'postgresql',
          status_protocol => 'postgresql'},
      ],
      'Puppet_enterprise::Profile::Ace_server' =>  [
        { type => 'ace',
          port_param     => 'ssl_listen_port',
          service_prefix => '',
          status_prefix => 'admin/status',
          status_key     => 'ace-server',
          display_name   => 'Agentless Catalog Executor Service'},
      ],
      'Puppet_enterprise::Profile::Bolt_server' =>  [
        { type => 'bolt',
          port_param     => 'ssl_listen_port',
          service_prefix => '',
          status_prefix => 'admin/status',
          status_key     => 'bolt-server',
          display_name   => 'Bolt Service'},
      ],
    }

    $query_or_clause =
      pe_concat(['or'],
                pe_keys($classes_to_fetch).map |$title| {['=', 'title', $title]})

    $rows = puppetdb_query(
      ['from', 'resources',
        ['extract', ['certname', 'title', 'parameters', 'tags'],
          ['and',
            ['=', ['node','active'], true],
            ['=', 'type', 'Class'],
            $query_or_clause]],
        ['order_by', [['certname','asc'], ['title', 'asc']]]]
    )

    #################
    # Services

    $queried_services_nested = $rows.map |$row| {
      $resource_title = $row['title']
      $class_parameters = $row['parameters']
      $node_certname = $row['certname']

      $service_defs_for_class = $classes_to_fetch[$resource_title]
      $service_defs_for_class.map |$service_def| {
        $service_prefix = $service_def['service_prefix']
        $service_type = $service_def['type']
        $service_display_name = $service_def['display_name']
        $service_status_key = $service_def['status_key']
        $service_port_param = $service_def['port_param']
        $status_port_param = $service_def['status_port_param']
        $service_disabled_param = $service_def['disabled_param']
        $service_port = pe_pick($class_parameters[$service_port_param], $service_def['port_default'])
        $service_protocol = pe_pick($service_def['protocol'], 'https')

        # Here we use the status port param if it is set, but
        # default to the service port if it is not set
        $status_port = pe_pick($class_parameters[$status_port_param], $service_port)
        $status_protocol = pe_pick($service_def['status_protocol'], $service_protocol)
        $pg_certs_dir = "${puppet_enterprise::server_data_dir}/postgresql/${puppet_enterprise::params::postgres_version}/data/certs"

        $status_prefix = ($service_def['status_prefix'] =~ Undef) ? {
          true    => 'status/v1/services', # The trapper-keeper status endpoint.
          default => $service_def['status_prefix'], # For exceptional cases (Puma/Postgresql)
        }
        $status_url    = "${status_protocol}://${node_certname}:${status_port}/${status_prefix}"

        case $service_type {
          'postgresql': {
            $additional_service_parameters = {
              'database_configs'            => {
                'activity'     => {
                  'database' => $class_parameters['activity_database_name'],
                  'user'     => $class_parameters['activity_database_super_user'],
                },
                'classifier'   => {
                  'database' => $class_parameters['classifier_database_name'],
                  'user'     => $class_parameters['classifier_database_super_user'],
                },
                'inventory'    => {
                  'database' => $class_parameters['inventory_database_name'],
                  'user'     => $class_parameters['inventory_database_super_user'],
                },
                'orchestrator' => {
                  'database' => $class_parameters['orchestrator_database_name'],
                  'user'     => $class_parameters['orchestrator_database_super_user'],
                },
                'rbac'         => {
                  'database' => $class_parameters['rbac_database_name'],
                  'user'     => $class_parameters['rbac_database_super_user'],
                },
                'pe-hac'       => {
                  'database' => $class_parameters['host_action_collector_database_name'],
                  'user'     => $class_parameters['host_action_collector_database_super_user'],
                },
                'pe-patching'       => {
                  'database' => $class_parameters['patching_database_name'],
                  'user'     => $class_parameters['patching_database_super_user'],
                },
                'pe-infra-assistant'       => {
                  'database' => $class_parameters['infra_assistant_database_name'],
                  'user'     => $class_parameters['infra_assistant_database_super_user'],
                },
                'pe-workflow' => {
                  'database' => $class_parameters['workflow_database_name'],
                  'user'     => $class_parameters['workflow_database_super_user'],
                },
              },
              'replication_mode'            => $class_parameters['replication_mode'],
              'replication_source_hostname' => $class_parameters['replication_source_hostname'],
              'certs_dir'                   => $pg_certs_dir,
            }
          }
          default: {
            $additional_service_parameters = {}
          }
        }

        $exclude_service = ($service_disabled_param and $class_parameters[$service_disabled_param])

        unless $exclude_service  {
          {
            type          => $service_type,
            url           => "${service_protocol}://${node_certname}:${service_port}/${service_prefix}",
            server        => $node_certname,
            port          => $service_port,
            prefix        => $service_prefix,
            status_url    => $status_url,
            status_prefix => $status_prefix,
            status_key    => $service_status_key,
            node_certname => $node_certname,
            display_name  => $service_display_name,
          } + $additional_service_parameters
        }
      }
    }

    $queried_services = pe_flatten($queried_services_nested).filter |$elem| { $elem != undef}

    #################
    # Nodes

    $master_node = {
      'role' => 'primary_master',
      'display_name' => 'Primary',
      'certname' => $puppet_enterprise::puppet_master_host,
      'order' => 1,
    }

    $rows_with_replica_role_tags = $rows.filter |$row| {
      $replica_role = 'puppet_enterprise::profile::primary_master_replica'
      $row['tags'].filter |$tag| { $tag == $replica_role }[0] != undef
    }
    $replicas = pe_unique($rows_with_replica_role_tags.map |$row| { $row['certname'] })
    $replica_nodes = $replicas.map |$certname| {
      {
        'role' => 'primary_master_replica',
        'display_name' => 'Replica',
        'certname' => $certname,
        'order' => 2,
      }
    }

    $master_and_replicas = [$master_node['certname']] + $replicas

    $database_node_rows = $rows.filter |$row| {
      (
        $row['title'] == 'puppet_enterprise::profile::database' and
        !($row['certname'] in $master_and_replicas)
      )
    }
    $database_nodes = pe_unique($database_node_rows.map |$row| {
      {
        'role' => 'database',
        'display_name' => 'Database',
        'certname' => $row['certname'],
        'order' => 3,
      }
    })

    $compiler_node_rows = $rows.filter |$row| {
      (
        $row['title'] == 'puppet_enterprise::profile::master' and
        !($row['certname'] in $master_and_replicas)
      )
    }
    $compiler_nodes = pe_unique($compiler_node_rows.map |$row| {
      {
        'role' => 'compiler',
        'display_name' => 'Compiler',
        'certname' => $row['certname'],
        'order' => 4,
      }
    })

    $queried_nodes = pe_concat(
      [ $master_node ],
      $replica_nodes,
      $database_nodes,
      $compiler_nodes,
    )
  }
  else {
    $queried_services = []
    $queried_nodes = []
  }

  Pe_hocon_setting {
    ensure  => present,
    path    => $path,
  }

  file { $path:
    ensure => present,
    owner  => $user,
    group  => $group,
    mode   => $mode,
  }

  pe_hocon_setting { "${path}/services":
    setting => 'services',
    value   => pe_concat($queried_services, $additional_services),
    type    => 'array',
  }

  pe_hocon_setting { "${path}/nodes":
    setting => 'nodes',
    value   => pe_concat($queried_nodes, $additional_nodes),
    type    => 'array',
  }

  pe_hocon_setting { "${path}/certs":
    setting => 'certs',
    value   => {
      'ca-cert' => $puppet_enterprise::params::localcacert,
    },
    type    => 'hash',
  }
}
