define puppet_enterprise::pg::pglogical::subscription(
  String $subscription_name,
  String $database,
  String $host = '',
  String $user = '',
  String $ssl_cert_file = '',
  String $ssl_key_file = '',
  String $ssl_ca_file = '',
  Enum[present, absent] $ensure = present,
  Integer $keepalives_idle = $puppet_enterprise::pglogical_keepalives_idle,
  Optional[Integer] $keepalives_interval = $puppet_enterprise::pglogical_keepalives_interval,
  Integer $keepalives_count = $puppet_enterprise::pglogical_keepalives_count,
  Boolean $synchronize_structure = true,
  Boolean $enabled = true,
) {
  # lint:ignore:case_without_default
  case $ensure {
    present: {
      $provider_dsn_hash = {
        'host'                => $host,
        'port'                => $pe_postgresql::params::port,
        'dbname'              => $database,
        'user'                => $user,
        'sslmode'             => 'verify-full',
        'sslcert'             => $ssl_cert_file,
        'sslkey'              => $ssl_key_file,
        'sslrootcert'         => $ssl_ca_file,
        'keepalives_idle'     => $keepalives_idle,
        'keepalives_interval' => $keepalives_interval,
        'keepalives_count'    => $keepalives_count,
      }

      $provider_dsn = pe_hash2dsn($provider_dsn_hash)

      if $enabled {
        # https://github.com/2ndQuadrant/pglogical/blob/REL2_x_STABLE/docs/README.md

        # PE-38247: The "synchronize" parameters only apply to the initial subscription initialization.
        # What we're doing here will wait for the entire database to get synchronized over before
        # proceeding. This may take a while for a large pe-activity database (a 12GB database
        # was tested to take about 10 minutes).
        #
        # Another strategy would be to create the subscription with synchronize_structure = true and
        # synchronize_data = false, wait for sync, drop the subscription, then create one with
        # synchronize_structure = false, synchronize_data = true and let it do the initial data
        # sync in the background. However, because this data sync may take a while, this results
        # in "puppet infra status" showing services in a failed or unknown state due to
        # pg_stat_subscription showing the subscription in the "startup" state instead of "streaming".
        # This may cause users to run "puppet infra reinitialize replica" to try to fix it, which
        # just starts the process over. Also, our services assume the database is good once we
        # exit this part of the code, so we take the time hit here.
        puppet_enterprise::psql { "pglogical_subscription ${title} create-sql":
          db      => $database,
          command => "SELECT pglogical.create_subscription(
                        subscription_name := '${subscription_name}',
                        provider_dsn := '${provider_dsn}',
                        synchronize_structure := ${synchronize_structure})",
          unless  => "SELECT * from pglogical.subscription WHERE sub_name='${subscription_name}'",
        }
        ~> puppet_enterprise::psql { "pglogical_subscription ${title} wait-for-sync":
          db          => $database,
          command     => "SELECT pglogical.wait_for_subscription_sync_complete('${subscription_name}')",
          refreshonly => true,
        }
        
        # create_subscription should enable it automatically, but check it here and enable it if not
        puppet_enterprise::psql { "pglogical_subscription ${title} enable-sql":
          db      => $database,
          command => "SELECT pglogical.alter_subscription_enable('${subscription_name}')",
          unless  => "SELECT * from pglogical.subscription WHERE sub_name='${subscription_name}' and sub_enabled = true",
          require => Puppet_enterprise::Psql["pglogical_subscription ${title} create-sql"],
        }
      } else {
        # The union on the unless clause ensures we don't try to disable when no
        # subscription exists yet.
        puppet_enterprise::psql { "pglogical_subscription ${title} disable-sql":
          db      => $database,
          command => "SELECT pglogical.alter_subscription_disable('${subscription_name}')",
          unless  => "SELECT sub_name from pglogical.subscription
                      WHERE sub_name='${subscription_name}' and sub_enabled = false
                      UNION
                      SELECT 'no-subscription'
                      WHERE not exists (SELECT * from pglogical.subscription
                                        WHERE sub_name = '${subscription_name}')",
        }
      }
    }

    absent: {
      puppet_enterprise::psql { "pglogical_subscription ${title} drop-sql":
        db      => $database,
        command => "SELECT pglogical.drop_subscription('${subscription_name}', false)",
        # query that returns rows if the subscription doesn't exist
        unless  => "SELECT *
                    FROM (SELECT COUNT(*)
                          FROM pglogical.subscription
                          WHERE sub_name='${subscription_name}') c
                    WHERE c.count = 0",
      }
    }
  }
  # lint:endignore
}
