# This class sets up the agent. For more information, see the [README.md](./README.md)
#
# @param manage_symlinks [Boolean] Flag to enable creation of convenience links
# @param pxp_enabled [Boolean] Flag to enable the pxp-agent service
# @param pcp_broker_host [String] Certname of the PCP broker
# @param pcp_broker_port [Integer] Port used by the PCP broker
# @param manage_puppet_conf [Boolean] Flag for noting if we manage various parameters defined by
#                                     this class in puppet.conf. When set to false, parameters such
#                                     as crl_refresh_interval, exclude_unchanged_resources, strict,
#                                     and preprocess_deferred will have no effect.
# @param server_list [Array[String]] An array of certnames running puppetserver for the agent
#                                    to point to when requesting a catalog.
# @param pcp_broker_ws_uris [Array[String]] *Deprecated* A list of PCP broker websocket URIs
#                                           (i.e. wss://) for pxp-agent to communicate with.
#                                           pcp_broker_list should be used instead.
# @param pcp_broker_list [Array[String]] A list of PCP brokers for pxp-agent to communicate with.
#                                        Will default to $pcp_broker_host:$pcp_broker_port when not set.
# @param $package_inventory_enabled [Boolean] Determines if we drop the file to enable the package inventory.
# @param $master_uris [Optional[Array[String]]] *Deprecated* Use primary_uris instead.
# @param $primary_uris [Optional[Array[String]] An array of primary servers for pxp-agent to point to. When
#                                               unset, will default to the puppet_master_host.
# @param crl_refresh_interval [Optional[String]] A duration string indicating how often the agent should fetch
#                                                an updated CRL. When unset, the CRL is never refreshed.
# @param exclude_unchanged_resources [Optional[Boolean]] Whether to omit data about unchanged resources from
#                                                        reports. The underlying Puppet default for this setting is
#                                                        false in 7.x and true in 8.x.
# @param strict [Optional[Enum]] Set strict mode. See https://github.com/puppetlabs/puppet/blob/90720bd26db416c89371f9b5e4bab7b1655a9fa7/lib/puppet/defaults.rb#L187-L190
# @param preprocess_deferred [Optional[Boolean]] Set preprocess_deferred on or off.  See 
#                                                See https://github.com/puppetlabs/puppet/blob/90720bd26db416c89371f9b5e4bab7b1655a9fa7/lib/puppet/defaults.rb#L2098-L2100
class puppet_enterprise::profile::agent(
  Boolean       $manage_symlinks                  = $puppet_enterprise::manage_symlinks,
  Boolean       $pxp_enabled                      = true,
  String        $pcp_broker_host                  = $puppet_enterprise::pcp_broker_host,
  Integer       $pcp_broker_port                  = $puppet_enterprise::pcp_broker_port,
  Boolean       $manage_puppet_conf               = true,
  Array[String] $server_list                      = [],
  Optional[Array[String]] $pcp_broker_ws_uris     = undef,
  Array[String] $pcp_broker_list                  = [],
  Boolean       $package_inventory_enabled        = false,
  Optional[Array[String]] $master_uris            = undef,
  Optional[Array[String]] $primary_uris           = $master_uris,
  Optional[String] $crl_refresh_interval          = undef,
  Optional[Boolean] $exclude_unchanged_resources  = undef,
  Optional[Enum['off','warning','error']] $strict = undef,
  Optional[Boolean] $preprocess_deferred          = undef,
) inherits puppet_enterprise {
  puppet_enterprise::deprecated_parameter{ 'puppet_enterprise::profile::agent::master_uris':
    replacement         => 'puppet_enterprise::profile::agent::primary_uris',
    suppress_on_install => true,
  }
  puppet_enterprise::deprecated_parameter{ 'puppet_enterprise::profile::agent::pcp_broker_ws_uris':
    replacement         => 'puppet_enterprise::profile::agent::pcp_broker_list',
    suppress_on_install => true,
  }

  include puppet_enterprise::symlinks

  if $manage_symlinks {
    File <| tag == 'pe-agent-symlinks' |>
  }

  $_identity = pe_empty($facts['identity']) ? {
    false  => $facts['identity'],
    true => {},
  }

  # We still manage older agent installs, so if identity isn't specified fallback to only enabling
  # the pxp-agent service.
  if ($puppet_enterprise::params::pxp_compatible and
     ($_identity['privileged'] or $pxp_enabled)) {

    # if pcp_broker_ws_uris is set we can't configure pxp to use version 2.
    if (pe_empty($pcp_broker_ws_uris) and $puppet_enterprise::params::pcp_v2_compatible) {
       $pcp_version = '2'
    } else {
       $pcp_version = undef
    }

    $pcp_endpoint = if $pcp_version == '2' { 'pcp2' } else { 'pcp' }

    if pe_size($pcp_broker_list) > 0 {
      if !pe_empty($pcp_broker_ws_uris) {
        warning('Both $pcp_broker_ws_uris and $pcp_broker_list are set. $pcp_broker_ws_uris will be ignored.')
      }
      $pcp_broker_ws_uris_ = $pcp_broker_list.map |$broker| { "wss://${broker}/${pcp_endpoint}/" }
    } elsif !pe_empty($pcp_broker_ws_uris) {
        $pcp_broker_ws_uris_ = $pcp_broker_ws_uris
    } else {
      $pcp_broker_ws_uris_ = ["wss://${pcp_broker_host}:${pcp_broker_port}/${pcp_endpoint}/"]
    }

    if pe_empty($primary_uris) {
      if pe_size($server_list) > 0 {
        $_primary_uris = $server_list
      } else {
        $_primary_uris = ["https://${puppet_enterprise::puppet_master_host}:${puppet_enterprise::puppet_master_port}"]
      }
    } else {
      $_primary_uris = $primary_uris
    }

    class { 'puppet_enterprise::pxp_agent':
      broker_ws_uri => $pcp_broker_ws_uris_,
      primary_uris   => $_primary_uris,
      pcp_version   => $pcp_version,
      enabled       => $pxp_enabled,
    }
  }

  Pe_ini_setting {
    ensure  => present,
    path    => "${::puppet_enterprise::params::confdir}/puppet.conf",
    section => 'main',
  }

  if $manage_puppet_conf {
    # Some of these used to be set in agent, so we ensure they are no longer
    # there since we want them all in main now.
    $settings = ['crl_refresh_interval', 'exclude_unchanged_resources']
    $settings.each |$s| {
      pe_ini_setting { "puppet.conf remove ${s} from agent section":
        ensure  => absent,
        section => 'agent',
        setting => $s,
      }
    }

    if $puppet_enterprise::params::agent_failover_compatible and pe_size($server_list) > 0 {
      # (PE-37029) We are leaving this one in the 'agent' section rather than main, since
      # pe-backup-tools and pe-opsworks-tools are hardcoded to override the value in
      # the agent section.
      pe_ini_setting { 'puppet.conf server_list':
        setting => 'server_list',
        section => 'agent',
        value   => pe_join($server_list, ','),
      }
    }

    if $crl_refresh_interval !~ Undef {
      # Don't configure this on PE server nodes; these have their own ways
      # of syncing the full CRL, whereas this feature will always fetch
      # the infra CRL.
      if ($facts['pe_server_version'] == undef) {
        pe_ini_setting { 'puppet.conf crl_refresh_interval':
          setting => 'crl_refresh_interval',
          value   => $crl_refresh_interval,
        }
      } else {
        pe_ini_setting { 'puppet.conf crl_refresh_interval':
          ensure  => absent,
          setting => 'crl_refresh_interval',
        }
      }
    }

    if $exclude_unchanged_resources !~ Undef {
      pe_ini_setting { 'puppet.conf exclude_unchanged_resources':,
        setting => 'exclude_unchanged_resources',
        value   => $exclude_unchanged_resources,
      }
    }

    if $strict !~ Undef {
      pe_ini_setting { 'puppet.conf strict':
        setting => 'strict',
        value   => $strict,
      }
    }

    if $preprocess_deferred !~ Undef {
      pe_ini_setting { 'puppet.conf preprocess_deferred':
        setting => 'preprocess_deferred',
        value   => $preprocess_deferred,
      }
    }
  }

  # Non-root users can't stat /opt/puppetlabs/puppet/cache
  # Note: The privileged flag of the identity fact only exists in Facter 3.2+
  # (puppet-agent 1.5.3, PE 2016.2.1), so an agent of that antiquity would need
  # to upgrade before package inventory data would be managed.
  if ($_identity['privileged']) {
    if $package_inventory_enabled {
      file {
        $puppet_enterprise::params::package_inventory_enabled_file:
          ensure => present,
          mode   => '0664',
      }
    } else {
      file {
        $puppet_enterprise::params::package_inventory_enabled_file:
          ensure => absent,
      }
    }
  }

  if !($facts['kernel'] in [ 'windows', 'Darwin' ]) {
    $uninstaller_path  = '/opt/puppetlabs/bin/puppet-enterprise-uninstaller'
    $_uninstall_script = file($uninstaller_path, '/dev/null')

    if $_uninstall_script =~ String[1] {
      file { $uninstaller_path :
        ensure  => file,
        mode    => '0544',
        content => $_uninstall_script,
      }
    }
  }
}
