# frozen_string_literal: true

require 'json'
require 'time'
require 'rexml/document'
require 'logger'
require 'puppet_x/puppetlabs/netconf/client'
require 'puppet_x/puppetlabs/netconf/redactor'

# Session provides a simplified API for writing NETCONF tasks in Bolt.
#
# This module handles all the boilerplate of establishing NETCONF connections,
# managing sessions, tracking operations, and formatting results. Tasks can
# focus on the NETCONF operations themselves rather than connection management.
#
# @example Basic usage in a task
#   require 'puppet_x/puppetlabs/netconf/session'
#
#   PuppetX::Puppetlabs::Netconf::Session.with_session do |session|
#     session.get_config('running')
#   end
#
# @example Making configuration changes
#   PuppetX::Puppetlabs::Netconf::Session.with_session do |session|
#     # Default uses target_datastore='candidate'
#     session.lock
#     session.edit_config(config_xml)
#     session.commit
#     session.unlock
#   end
#
# @example Using noop mode for dry runs
#   PuppetX::Puppetlabs::Netconf::Session.with_session(noop: true) do |session|
#     # Operations are simulated, not executed
#     session.edit_config(config_xml)
#     session.commit
#   end
#
# @example Accessing task parameters in the session
#   PuppetX::Puppetlabs::Netconf::Session.with_session do |session|
#     # Task parameters are available via session.task_params
#     if session.task_params['backup_config']
#       backup = session.get_config('running')
#       # Save backup...
#     end
#   end
#
module PuppetX
  module Puppetlabs
    module Netconf
      # Module for managing NETCONF sessions in Bolt tasks.
      # @since 1.0.0
      module Session
        # Execute a NETCONF session with automatic connection management.
        #
        # This method:
        # - Reads task input from STDIN (including _target connection info)
        # - Establishes a NETCONF connection using the target parameters
        # - Yields a session object for executing NETCONF operations
        # - Automatically generates a detailed session report
        # - Handles all errors and formats them for Bolt
        # - Ensures the connection is properly closed
        #
        # @param source_datastore [String] Default source datastore for read operations ('running', 'candidate', 'startup')
        # @param target_datastore [String] Default target datastore for write operations ('running', 'candidate', 'startup')
        # @param options [Hash] Additional session options that can include:
        #   - 'noop' [Boolean]: Run in no-operation mode (simulate changes without applying)
        #   - Any other options are passed through to the session
        #
        # @yield [session] Block where NETCONF operations are performed
        # @yieldparam session [SessionWrapper] Session object with NETCONF operations
        #
        # @note This method reads from STDIN and writes to STDOUT, then exits.
        #       It is designed to be the main entry point for NETCONF tasks.
        #
        # @example Task implementation with default datastores
        #   #!/usr/bin/env ruby
        #   require 'puppet_x/puppetlabs/netconf/session'
        #
        #   PuppetX::Puppetlabs::Netconf::Session.with_session do |session|
        #     # Uses source_datastore='running' and target_datastore='candidate' by default
        #     config = session.get_config  # Gets from 'running'
        #     session.lock                 # Locks 'candidate'
        #     session.edit_config(config)  # Edits 'candidate'
        #     session.commit
        #     session.unlock              # Unlocks 'candidate'
        #   end
        #
        # @example Task implementation with custom datastores
        #   PuppetX::Puppetlabs::Netconf::Session.with_session(target_datastore: 'running') do |session|
        #     # Direct edits to running datastore
        #     session.edit_config(config)  # Edits 'running' directly
        #   end
        #
        def self.with_session(target: nil,
                              source_datastore: nil, target_datastore: nil,
                              noop: nil,
                              redact_patterns: nil,
                              **options)
          # Always parse task parameters from STDIN
          # When called from Bolt, STDIN will always have the task parameters
          stdin_params = begin
                           JSON.parse(STDIN.read)
                         rescue => e
                           # If anything goes wrong reading STDIN, just use empty params
                           {}
                         end

          # Extract meta-parameters from STDIN
          # Direct parameters to with_session take precedence over STDIN
          target ||= stdin_params['_target']
          noop ||= stdin_params['_noop']
          log_level = options[:log_level] || stdin_params['_bolt_log_level']

          # For datastore and redaction params, with_session params take precedence
          # This allows task implementations to enforce specific datastores
          source_datastore ||= stdin_params['source_datastore']
          target_datastore ||= stdin_params['target_datastore']
          redact_patterns ||= stdin_params['redact_patterns']

          # Extract connection info from target (required)
          raise ArgumentError, 'target is required (provide target parameter or _target from STDIN)' unless target

          host = target['host']
          user = target['user']
          password = target['password']
          port = target['port']
          private_key = target['private-key']

          # Extract timeout settings from target
          # Global timeout serves as default for all other timeouts
          timeout = target['timeout'] || 10
          connect_timeout = target['connect-timeout']
          disconnect_timeout = target['disconnect-timeout']
          command_timeout = target['command-timeout']
          hello_timeout = target['hello-timeout']
          idle_timeout = target['idle-timeout']
          keepalive_interval = target['keepalive-interval']

          # Extract reconnection settings from target
          auto_reconnect = target['auto-reconnect']
          reconnect_attempts = target['reconnect-attempts']
          reconnect_delay = target['reconnect-delay']

          # Extract host key settings from target
          host_key_check = target['host-key-check']

          # Set defaults
          port ||= 830
          source_datastore ||= 'running'
          target_datastore ||= 'running'

          # Ensure we have required connection parameters
          raise ArgumentError, 'host is required' unless host
          raise ArgumentError, 'user is required' unless user
          raise ArgumentError, 'password or private-key is required' unless password || private_key

          # Configure logging with the resolved log level (from stdin or override)
          configure_logging(log_level)

          # Separate task parameters from meta-parameters
          # Exclude Bolt's special underscore parameters (_target, _noop, _bolt_log_level, etc.)
          stdin_task_params = stdin_params.reject { |k, _| k.start_with?('_') }

          # Build session options with correct precedence:
          # with_session options take precedence over STDIN task params
          # This allows task implementations to enforce specific behavior
          session_options = stdin_task_params.merge(options)

          # Add the resolved control parameters
          session_options['source_datastore'] = source_datastore
          session_options['target_datastore'] = target_datastore
          session_options['_noop'] = noop if noop
          session_options['redact_patterns'] = redact_patterns if redact_patterns
          session_options['include_default_patterns'] = options[:include_default_patterns] if options.key?(:include_default_patterns)

          # Build client options
          client_options = {
            port: port,
            timeout: timeout
          }

          # Add authentication
          if private_key
            client_options[:private_key] = private_key
          else
            client_options[:password] = password
          end

          # Add optional parameters only if specified
          client_options[:connect_timeout] = connect_timeout if connect_timeout
          client_options[:disconnect_timeout] = disconnect_timeout if disconnect_timeout
          client_options[:command_timeout] = command_timeout if command_timeout
          client_options[:hello_timeout] = hello_timeout if hello_timeout
          client_options[:idle_timeout] = idle_timeout if idle_timeout
          client_options[:keepalive_interval] = keepalive_interval if keepalive_interval

          client_options[:auto_reconnect] = auto_reconnect if auto_reconnect
          client_options[:reconnect_attempts] = reconnect_attempts if reconnect_attempts
          client_options[:reconnect_delay] = reconnect_delay if reconnect_delay

          # Handle host key verification
          if host_key_check == false
            client_options[:verify_host_key] = :never
          elsif host_key_check == true
            client_options[:verify_host_key] = :always
          end

          # Create a minimal target info object for session reporting
          # This provides just the connection details needed for the session wrapper
          target_info = TargetInfo.new(host, user, port)

          # Handle redaction configuration
          include_defaults = session_options.fetch('include_default_patterns', true)

          redactor = if redact_patterns&.any?
                       # User provided custom patterns
                       PuppetX::Puppetlabs::Netconf::Redactor.new(redact_patterns, include_defaults: include_defaults)
                     elsif include_defaults
                       # No custom patterns but defaults are enabled (default behavior)
                       PuppetX::Puppetlabs::Netconf::Redactor.new
                     else
                       # User explicitly disabled defaults and provided no patterns - no redaction
                       nil
                     end

          # Pass redactor to client if one was created
          client_options[:redactor] = redactor if redactor

          # Establish NETCONF connection
          client = PuppetX::Puppetlabs::Netconf::Client.new(host, user, client_options)
          client.connect

          # Create session wrapper that tracks operations
          session = SessionWrapper.new(client, target_info, session_options)

          begin
            # Execute the user's block with the session
            result = yield session

            # Generate and output the session report
            report = session.generate_report

            # If the block returned a result, merge it with the report
            if result&.is_a?(Hash)
              report.merge!(result)
            elsif result
              report['result'] = result
            end

            # Return the report - tasks will handle output
            report
          rescue => e
            # Generate error report with session details
            report = session.generate_report

            # Redact error message if redactor is enabled
            error_message = session.redactor ? session.redactor.redact_string(e.message) : e.message

            report['error'] = {
              'message' => error_message,
              'type' => e.class.name,
              'backtrace' => e.backtrace[0..5]
            }

            # Re-raise the error - let tasks handle it
            raise
          ensure
            # Always disconnect, even if an error occurred
            # Note: We do NOT auto-discard candidate changes here because:
            # 1. Some vendors (Juniper) use private candidate that auto-cleans
            # 2. Some workflows may want to preserve uncommitted changes across sessions
            # 3. It's safer to let users explicitly manage their candidate changes
            #
            # If users experience stuck locks (like with Arista), they should either:
            # - Always use proper lock/unlock workflow
            # - Explicitly call discard-changes when done
            # - Use the running datastore instead of candidate
            client.disconnect if client&.open?
          end
        end

        # Configure logging based on the specified log level
        #
        # @param log_level [String, nil] The desired log level (debug, info, warn, error, etc.)
        # @return [Logger] The configured logger instance
        # @api private
        def self.configure_logging(log_level = nil)
          # Create a logger that writes to STDERR
          logger = Logger.new(STDERR)

          # Set formatter to match Bolt's format
          logger.formatter = proc do |severity, datetime, progname, msg|
            "#{datetime.strftime('%Y-%m-%dT%H:%M:%S.%6N')} %-6s [#{progname || 'PuppetX::Puppetlabs::Netconf'}] #{msg}\n" % severity
          end

          if log_level
            # Convert to appropriate Logger level
            logger.level = case log_level.to_s.downcase
                           when 'trace', 'debug'
                             Logger::DEBUG
                           when 'info'
                             Logger::INFO
                           when 'warn', 'warning'
                             Logger::WARN
                           when 'error'
                             Logger::ERROR
                           when 'fatal'
                             Logger::FATAL
                           else
                             # Default to WARN if invalid level provided
                             Logger::WARN
                           end

            logger.debug("Configured logging with level: #{log_level}")
          else
            # Default to WARN if no level specified
            logger.level = Logger::WARN
          end

          # Add trace method if it doesn't exist (for Ruby's Logger)
          unless logger.respond_to?(:trace)
            def logger.trace(msg)
              debug(msg) if level <= Logger::DEBUG
            end
          end

          # Store logger in module attribute for access
          @logger = logger
        end

        # Get the configured logger instance
        # @return [Logger] The logger instance
        def self.logger
          @logger ||= begin
            # Create a default logger if not configured
            logger = Logger.new(STDERR)
            logger.level = Logger::WARN
            logger.formatter = proc do |severity, datetime, progname, msg|
              "#{datetime.strftime('%Y-%m-%dT%H:%M:%S.%6N')} %-6s [#{progname || 'PuppetX::Puppetlabs::Netconf'}] #{msg}\n" % severity
            end

            # Add trace method if it doesn't exist
            unless logger.respond_to?(:trace)
              def logger.trace(msg)
                debug(msg) if level <= Logger::DEBUG
              end
            end

            logger
          end
        end

        # Internal class that holds target connection information for session reporting
        # @api private
        class TargetInfo
          attr_reader :host, :user, :port

          def initialize(host, user, port)
            @host = host
            @user = user
            @port = port
          end

          # Returns a safe string representation of the target
          # @return [String] Target identifier in format user@host:port
          def safe_name
            "#{@user}@#{@host}:#{@port}"
          end
        end

        # SessionWrapper provides session management for NETCONF operations.
        #
        # This class wraps a NETCONF connection and provides:
        # - Operation tracking with timestamps and status
        # - Support for noop (dry-run) mode
        # - Automatic session report generation
        # - Configuration change tracking (when before/after states are available)
        # - Optional redaction of sensitive data
        #
        # @api public
        class SessionWrapper
          attr_reader :connection, :target, :operations, :start_time, :noop, :source_datastore, :target_datastore, :task_params, :logger, :redactor

          def initialize(connection, target, options = {})
            @connection = connection
            @target = target
            @operations = []
            @start_time = Time.now
            @noop = options['_noop'] || false
            @source_datastore = options['source_datastore'] || 'running'
            @target_datastore = options['target_datastore'] || 'running'
            @before_config = nil
            @after_config = nil
            # Use the same redactor instance as the client
            @redactor = @connection.redactor
            # Store task parameters (excluding Bolt's underscore parameters and session control params)
            # Note: Connection params (host, user, port, password) come from _target hash, not top-level
            meta_params = ['source_datastore', 'target_datastore', 'redact_patterns']
            @task_params = options.reject do |k, _|
              k_str = k.to_s
              k_str.start_with?('_') || meta_params.include?(k_str)
            end
            # Use the logger configured by the session
            @logger = Session.logger
            # Store custom results from user
            @custom_results = {}

            # Validate datastores against server capabilities
            validate_datastores unless @noop
          end

          # Retrieve configuration from a NETCONF datastore.
          #
          # @param filter [String, REXML::Element, nil] Optional subtree filter
          # @param source [String, nil] Optional source datastore override (defaults to session's source_datastore)
          # @return [Array<REXML::Element>] Configuration elements
          # @raise [NetconfError] If the operation fails
          def get_config(filter = nil, source: nil)
            source ||= @source_datastore
            operation = { 'operation' => 'get-config', 'source' => source, 'timestamp' => Time.now.iso8601 }
            result = @connection.rpc.get_config(source, filter)
            operation['status'] = 'success'

            # Warn if filter was provided but returned empty results
            if filter && !filter.to_s.strip.empty?
              data_elem = result.find { |elem| elem.name == 'data' }
              if data_elem && data_elem.elements.empty?
                @logger.warn("get_config filter returned no matching elements from #{source} datastore")
                @logger.debug("Filter used: #{filter.to_s[0..500]}") # Truncate large filters in debug
              end
            end

            @operations << operation
            result
          rescue => e
            operation['status'] = 'error'
            operation['error'] = e.message
            @operations << operation
            raise
          end

          # Edit configuration in a NETCONF datastore.
          #
          # @param config [String, REXML::Element, nil] Configuration to apply
          # @param target [String, nil] Optional target datastore override (defaults to session's target_datastore)
          # @return [Array<REXML::Element>] Response elements
          # @raise [NetconfError] If the operation fails
          def edit_config(config = nil, target: nil, &block)
            target ||= @target_datastore
            operation = { 'operation' => 'edit-config', 'target' => target, 'timestamp' => Time.now.iso8601 }

            # Capture before state if not already captured
            if @before_config.nil? && !@noop
              @before_config = get_config
            end

            if @noop
              operation['status'] = 'noop'
              operation['simulated'] = true
              @operations << operation
              return []
            end

            result = @connection.rpc.edit_config(target, config, &block)
            operation['status'] = 'success'
            @operations << operation

            # Capture after state for diff when editing running datastore directly
            if target == 'running' && @before_config
              @after_config = get_config
            end

            result
          rescue => e
            operation['status'] = 'error'
            operation['error'] = e.message
            @operations << operation
            raise
          end

          def commit
            operation = { 'operation' => 'commit', 'timestamp' => Time.now.iso8601 }

            # Check if device supports :candidate capability
            unless supports_candidate?
              error_msg = 'Device does not support :candidate capability required for commit operation'
              operation['status'] = 'error'
              operation['error'] = error_msg
              @operations << operation
              raise NetconfError::RpcError, error_msg
            end

            if @noop
              operation['status'] = 'noop'
              operation['simulated'] = true
              @operations << operation
              return []
            end

            result = @connection.rpc.commit
            operation['status'] = 'success'
            @operations << operation

            # Capture after state for diff
            if @target_datastore == 'candidate' && @before_config
              @after_config = get_config
            end

            result
          rescue NetconfError::RpcError
            # Re-raise our own errors
            raise
          rescue => e
            operation['status'] = 'error'
            operation['error'] = e.message
            @operations << operation
            raise
          end

          def lock(datastore = nil)
            target = datastore || @target_datastore
            operation = { 'operation' => 'lock', 'target' => target, 'timestamp' => Time.now.iso8601 }

            if @noop
              operation['status'] = 'noop'
              operation['simulated'] = true
              @operations << operation
              return []
            end

            result = @connection.rpc.lock(target)
            operation['status'] = 'success'
            @operations << operation
            result
          rescue => e
            operation['status'] = 'error'
            operation['error'] = e.message
            @operations << operation
            raise
          end

          def unlock(datastore = nil)
            target = datastore || @target_datastore
            operation = { 'operation' => 'unlock', 'target' => target, 'timestamp' => Time.now.iso8601 }

            if @noop
              operation['status'] = 'noop'
              operation['simulated'] = true
              @operations << operation
              return []
            end

            result = @connection.rpc.unlock(target)
            operation['status'] = 'success'
            @operations << operation
            result
          rescue => e
            operation['status'] = 'error'
            operation['error'] = e.message
            @operations << operation
            raise
          end

          def get(filter = nil, &block)
            operation = { 'operation' => 'get', 'timestamp' => Time.now.iso8601 }

            if @noop
              operation['status'] = 'noop'
              operation['simulated'] = true
              @operations << operation
              return []
            end

            result = if block
                       @connection.rpc.get(&block)
                     else
                       @connection.rpc.get(filter)
                     end

            # Warn if filter was provided but returned empty results
            if filter && !filter.to_s.strip.empty? && !block
              data_elem = result.find { |elem| elem.name == 'data' }
              if data_elem && data_elem.elements.empty?
                @logger.warn('get filter returned no matching elements')
                @logger.debug("Filter used: #{filter.to_s[0..500]}") # Truncate large filters in debug
              end
            end

            operation['status'] = 'success'
            @operations << operation
            result
          rescue => e
            operation['status'] = 'error'
            operation['error'] = e.message
            @operations << operation
            raise
          end

          # Retrieve a specific YANG schema from the device
          #
          # This method implements the standard NETCONF get-schema operation defined in RFC 6022.
          #
          # Standard NETCONF get-schema operation (RFC 6022 Section 3):
          #   <get-schema xmlns="urn:ietf:params:xml:ns:yang:ietf-netconf-monitoring">
          #     <identifier>module-name</identifier>
          #     <version>2023-02-07</version>  <!-- optional -->
          #     <format>yang</format>           <!-- optional, default: yang -->
          #   </get-schema>
          #
          # Response contains the schema text in a <data> element:
          #   <data xmlns="urn:ietf:params:xml:ns:yang:ietf-netconf-monitoring">
          #     module foo { ... }
          #   </data>
          #
          # @param identifier [String] The schema identifier/module name to retrieve
          # @param version [String, nil] The specific version/revision to retrieve (optional)
          # @param format [String] The schema format (default: 'yang', also supports 'yin')
          # @return [Array<REXML::Element>] Schema data elements
          # @raise [NetconfError::RpcError] If device doesn't support monitoring or schema not found
          def get_schema(identifier, version = nil, format = 'yang')
            operation = {
              'operation' => 'get-schema',
              'identifier' => identifier,
              'version' => version,
              'format' => format,
              'timestamp' => Time.now.iso8601
            }

            # Check if device supports ietf-netconf-monitoring (required for get-schema)
            monitoring_cap = @connection.server_capabilities.find { |c| c.include?('urn:ietf:params:xml:ns:yang:ietf-netconf-monitoring') }

            unless monitoring_cap
              error_msg = 'Device does not support ietf-netconf-monitoring capability required for get-schema operation'
              operation['status'] = 'error'
              operation['error'] = error_msg
              @operations << operation
              raise NetconfError::RpcError, error_msg
            end

            if @noop
              operation['status'] = 'noop'
              operation['simulated'] = true
              @operations << operation
              return []
            end

            result = @connection.rpc.get_schema(identifier, version, format)
            operation['status'] = 'success'
            @operations << operation
            result # Return the structured response like other operations
          rescue NetconfError::RpcError
            # Re-raise our own errors
            raise
          rescue => e
            operation['status'] = 'error'
            operation['error'] = e.message
            @operations << operation
            raise
          end

          # List all available schemas on the device
          #
          # This method implements the standard NETCONF schema discovery defined in RFC 6022.
          #
          # Standard Schema List Retrieval (RFC 6022 Section 2.4.4):
          # The client retrieves the list of supported schemas via a <get> operation:
          #   <get>
          #     <filter type="subtree">
          #       <netconf-state xmlns="urn:ietf:params:xml:ns:yang:ietf-netconf-monitoring">
          #         <schemas/>
          #       </netconf-state>
          #     </filter>
          #   </get>
          #
          # Expected Response Format:
          #   <netconf-state xmlns="urn:ietf:params:xml:ns:yang:ietf-netconf-monitoring">
          #     <schemas>
          #       <schema>
          #         <identifier>module-name</identifier>
          #         <version>2023-02-07</version>
          #         <format>yang</format>
          #         <namespace>http://example.com/namespace</namespace>
          #         <location>NETCONF</location>
          #       </schema>
          #       <!-- Additional schema entries -->
          #     </schemas>
          #   </netconf-state>
          #
          # Note: Some devices advertise ietf-netconf-monitoring capability but return empty schema lists
          # or don't implement the schemas subtree properly. This method will return an empty array
          # for such devices rather than attempting to extract information from capabilities.
          #
          # @return [Array<Hash>] Array of schema information hashes containing:
          #   - identifier: The schema/module name
          #   - version: The revision date
          #   - format: The schema format (yang, yin, etc.)
          #   - namespace: The XML namespace
          #   - location: Where the schema can be retrieved from
          def list_schemas
            operation = {
              'operation' => 'get',
              'filter' => 'netconf-state/schemas',
              'timestamp' => Time.now.iso8601
            }

            # Check if device supports ietf-netconf-monitoring
            monitoring_cap = @connection.server_capabilities.find { |c| c.include?('urn:ietf:params:xml:ns:yang:ietf-netconf-monitoring') }

            unless monitoring_cap
              error_msg = 'Device does not support ietf-netconf-monitoring capability required for schema listing'
              operation['status'] = 'error'
              operation['error'] = error_msg
              @operations << operation
              raise NetconfError::OperationNotSupported, error_msg
            end

            if @noop
              operation['status'] = 'noop'
              operation['simulated'] = true
              @operations << operation
              return []
            end

            # Try standard RFC 6022 approach first
            filter = <<-XML
    <netconf-state xmlns="urn:ietf:params:xml:ns:yang:ietf-netconf-monitoring">
      <schemas/>
    </netconf-state>
            XML

            begin
              result = get(filter)
              raw_response = result.map(&:to_s).join("\n")
            rescue PuppetX::Puppetlabs::Netconf::NetconfError::RpcError => e
              # If the device doesn't support netconf-state/schemas, provide a clear error
              # Common error tags that indicate lack of support:
              if e.message.downcase.include?('operation-failed') ||
                 e.message.downcase.include?('operation failed') ||
                 e.message.include?('invalid-value') ||
                 e.message.include?('unknown-namespace') ||
                 e.message.include?('bad-element') ||
                 e.message.include?('not supported')
                operation['status'] = 'error'
                operation['error'] = 'Device does not support netconf-state/schemas query'
                operation['details'] = e.message
                @operations << operation

                # Provide a more helpful error message
                error_msg = 'This device does not support the standard NETCONF schema listing operation (RFC 6022). '
                error_msg += "The device returned: #{e.message}. "
                error_msg += "Note: The device advertises ietf-netconf-monitoring capability but doesn't implement the schemas subtree. "
                error_msg += 'You may need to check device documentation for how to access YANG schemas.'

                raise PuppetX::Puppetlabs::Netconf::NetconfError::OperationNotSupported, error_msg
              end
              raise
            end

            # Check if device returned empty data
            if raw_response.strip == '<data/>' || raw_response.strip == '<data></data>'
              @logger.info('Device returned empty schema list')
              operation['status'] = 'success'
              operation['method'] = 'netconf-state'
              operation['count'] = 0
              @operations << operation
              return []
            end

            # Parse the standard response
            schemas = []
            result.each do |element|
              netconf_state = element.elements['netconf-state'] || element.elements['//netconf-state']
              next unless netconf_state
              schemas_elem = netconf_state.elements['schemas']
              next unless schemas_elem
              schemas_elem.elements.each('schema') do |schema|
                schema_info = {}
                schema.elements.each do |child|
                  case child.name
                  when 'identifier'
                    schema_info['identifier'] = child.text.strip
                  when 'version'
                    schema_info['version'] = child.text.strip
                  when 'format'
                    schema_info['format'] = child.text.strip
                  when 'namespace'
                    schema_info['namespace'] = child.text.strip
                  when 'location'
                    schema_info['location'] ||= []
                    schema_info['location'] << child.text.strip
                  end
                end
                schemas << schema_info if schema_info['identifier']
              end
            end

            operation['status'] = 'success'
            operation['method'] = 'netconf-state'
            operation['count'] = schemas.length
            @operations << operation
            schemas
          rescue => e
            operation['status'] = 'error'
            operation['error'] = e.message
            @operations << operation
            raise
          end

          # Extract schema information from device capabilities
          # This method parses module information from capability URLs
          # Note: This is separate from list_schemas as capabilities and schemas are different concepts
          #
          # @return [Array<Hash>] Array of module information extracted from capabilities
          def extract_modules_from_capabilities
            modules = []
            @connection.server_capabilities.each do |cap|
              # Parse module info from capability URLs
              # Format: http://example.com/namespace?module=name&revision=date
              next unless cap.include?('module=')
              parts = cap.split('?')
              next unless parts.length > 1
              params = {}
              parts[1].split('&').each do |param|
                key, value = param.split('=')
                params[key] = value if key && value
              end

              next unless params['module']
              module_info = {
                'identifier' => params['module'],
                'version' => params['revision'],
                'format' => params['format'] || 'yang',
                'namespace' => parts[0],
                'location' => 'NETCONF'
              }
              modules << module_info
            end
            modules
          end

          def discard_changes
            operation = { 'operation' => 'discard-changes', 'timestamp' => Time.now.iso8601 }

            # Check if device supports :candidate capability
            unless supports_candidate?
              error_msg = 'Device does not support :candidate capability required for discard-changes operation'
              operation['status'] = 'error'
              operation['error'] = error_msg
              @operations << operation
              raise NetconfError::RpcError, error_msg
            end

            if @noop
              operation['status'] = 'noop'
              operation['simulated'] = true
              @operations << operation
              return []
            end

            result = @connection.rpc.discard_changes
            operation['status'] = 'success'
            @operations << operation
            result
          rescue NetconfError::RpcError
            # Re-raise our own errors
            raise
          rescue => e
            operation['status'] = 'error'
            operation['error'] = e.message
            @operations << operation
            raise
          end

          def validate(source = nil)
            source ||= @target_datastore
            operation = {
              'operation' => 'validate',
              'source' => source,
              'timestamp' => Time.now.iso8601
            }

            # Check if device supports :validate capability
            unless supports_validate?
              error_msg = 'Device does not support :validate capability'
              operation['status'] = 'error'
              operation['error'] = error_msg
              @operations << operation
              raise NetconfError::RpcError, error_msg
            end

            if @noop
              operation['status'] = 'noop'
              operation['simulated'] = true
              @operations << operation
              return []
            end

            result = @connection.rpc.validate(source)
            operation['status'] = 'success'
            @operations << operation
            result
          rescue NetconfError::RpcError
            # Re-raise our own errors
            raise
          rescue => e
            operation['status'] = 'error'
            operation['error'] = e.message
            @operations << operation
            raise
          end

          def copy_config(source, target)
            operation = {
              'operation' => 'copy-config',
              'source' => source,
              'target' => target,
              'timestamp' => Time.now.iso8601
            }

            if @noop
              operation['status'] = 'noop'
              operation['simulated'] = true
              @operations << operation
              return []
            end

            result = @connection.rpc.copy_config(source, target)
            operation['status'] = 'success'
            @operations << operation
            result
          rescue => e
            operation['status'] = 'error'
            operation['error'] = e.message
            @operations << operation
            raise
          end

          def delete_config(target)
            operation = {
              'operation' => 'delete-config',
              'target' => target,
              'timestamp' => Time.now.iso8601
            }

            if target == 'running'
              error_msg = 'Cannot delete running configuration'
              operation['status'] = 'error'
              operation['error'] = error_msg
              @operations << operation
              raise NetconfError::RpcError, error_msg
            end

            if @noop
              operation['status'] = 'noop'
              operation['simulated'] = true
              @operations << operation
              return []
            end

            result = @connection.rpc.delete_config(target)
            operation['status'] = 'success'
            @operations << operation
            result
          rescue NetconfError::RpcError
            # Re-raise our own errors
            raise
          rescue => e
            operation['status'] = 'error'
            operation['error'] = e.message
            @operations << operation
            raise
          end

          def kill_session(session_id)
            operation = {
              'operation' => 'kill-session',
              'session_id' => session_id,
              'timestamp' => Time.now.iso8601
            }

            if @noop
              operation['status'] = 'noop'
              operation['simulated'] = true
              @operations << operation
              return []
            end

            result = @connection.rpc.kill_session(session_id)
            operation['status'] = 'success'
            @operations << operation
            result
          rescue => e
            operation['status'] = 'error'
            operation['error'] = e.message
            @operations << operation
            raise
          end

          # Convenience method to access server capabilities
          def server_capabilities
            @connection.server_capabilities
          end

          # Get the current session ID
          def session_id
            @connection.session_id
          end

          # Check if a specific capability is supported
          def supports_capability?(capability)
            server_capabilities.any? { |cap| cap.include?(capability) }
          end

          # Check if candidate configuration is supported
          def supports_candidate?
            supports_capability?(':candidate')
          end

          # Check if confirmed commit is supported
          def supports_confirmed_commit?
            supports_capability?(':confirmed-commit')
          end

          # Check if validate is supported
          def supports_validate?
            supports_capability?(':validate')
          end

          # Check if startup configuration is supported
          def supports_startup?
            supports_capability?(':startup')
          end

          # Check if rollback-on-error is supported
          def supports_rollback_on_error?
            supports_capability?(':rollback-on-error')
          end

          # Check if writable-running is supported
          def supports_writable_running?
            supports_capability?(':writable-running')
          end

          # Check if XPath filtering is supported
          def supports_xpath?
            supports_capability?(':xpath')
          end

          # Capture the after configuration state for diff generation
          # This should be called after making changes when using running datastore
          def capture_after_state
            return unless @before_config && @after_config.nil? && @target_datastore == 'running'
            @after_config = get_config
          end

          # Store custom results to be included in the session report
          #
          # This method allows tasks to add their own data to the final report.
          # The data will be included in a dedicated 'result' field to avoid
          # conflicts with session metadata.
          #
          # @param data [Hash] Custom data to include in the report
          # @example
          #   session.report_result({
          #     'status' => 'success',
          #     'interfaces' => interfaces,
          #     'count' => interfaces.length
          #   })
          def report_result(data)
            @custom_results.merge!(data)
          end

          # Generate session report
          def generate_report
            end_time = Time.now
            duration = end_time - @start_time

            # Always include diff report at top level
            diff_report = if @before_config && @after_config
                            # We have actual before/after states to compare
                            # Apply redaction to configs before generating diff if redactor is enabled
                            if @redactor
                              before_redacted = @redactor.redact_xml(@before_config)
                              after_redacted = @redactor.redact_xml(@after_config)
                              # Parse the redacted XML strings back to REXML for diff generation
                              before_doc = REXML::Document.new(before_redacted)
                              after_doc = REXML::Document.new(after_redacted)
                              generate_diff([before_doc.root], [after_doc.root])
                            else
                              generate_diff(@before_config, @after_config)
                            end
                          elsif @noop
                            # Changes were simulated in noop mode
                            {
                              'summary' => 'Changes simulated in noop mode',
                              'changes' => [],
                              'before_state_captured' => false,
                              'after_state_captured' => false
                            }
                          else
                            # No configuration changes were made (read-only operations)
                            {
                              'summary' => 'No configuration changes',
                              'changes' => [],
                              'before_state_captured' => !@before_config.nil?,
                              'after_state_captured' => !@after_config.nil?
                            }
                          end

            # Merge all report data at top level
            report = {
              'session' => {
                'target' => @target.safe_name,
                'user' => @target.user,
                'start_time' => @start_time.iso8601,
                'end_time' => end_time.iso8601,
                'duration' => duration.round(3),
                'source_datastore' => @source_datastore,
                'target_datastore' => @target_datastore,
                'noop' => @noop,
                'session_id' => @connection.session_id,
                'redaction_enabled' => !@redactor.nil?
              },
              'operations' => @operations,
              'metrics' => {
                'total_operations' => @operations.length,
                'successful_operations' => @operations.count { |op| op['status'] == 'success' },
                'failed_operations' => @operations.count { |op| op['status'] == 'error' },
                'noop_operations' => @operations.count { |op| op['status'] == 'noop' }
              }
            }.merge(diff_report)

            # Add custom results without redaction
            # (XML redaction patterns are for device data, not task results)
            unless @custom_results.empty?
              report['result'] = @custom_results
            end

            # Do not include task params in report - Bolt handles parameter logging at a higher level

            report
          end

          private

          # Load sensitive parameter names from task metadata

          # Validate that the configured datastores are supported by the device
          def validate_datastores
            capabilities = @connection.server_capabilities

            # Check source datastore
            unless datastore_supported?(@source_datastore, capabilities)
              raise StandardError, "Device #{@target.safe_name} does not support source datastore '#{@source_datastore}'. " \
                "Supported datastores: #{supported_datastores(capabilities).join(', ')}"
            end

            # Check target datastore
            return if datastore_supported?(@target_datastore, capabilities)
            raise StandardError, "Device #{@target.safe_name} does not support target datastore '#{@target_datastore}'. " \
              "Supported datastores: #{supported_datastores(capabilities).join(', ')}"
          end

          # Check if a specific datastore is supported based on capabilities
          def datastore_supported?(datastore, capabilities)
            # Running is always supported in NETCONF 1.0
            return true if datastore == 'running'

            # Check for specific datastore capabilities
            capability_map = {
              'candidate' => 'urn:ietf:params:netconf:capability:candidate:1.0',
              'startup' => 'urn:ietf:params:netconf:capability:startup:1.0'
            }

            required_capability = capability_map[datastore]
            return false unless required_capability

            capabilities.any? { |cap| cap.start_with?(required_capability) }
          end

          # Get list of supported datastores from capabilities
          def supported_datastores(capabilities)
            datastores = ['running'] # Always supported

            if capabilities.any? { |cap| cap.start_with?('urn:ietf:params:netconf:capability:candidate:1.0') }
              datastores << 'candidate'
            end

            if capabilities.any? { |cap| cap.start_with?('urn:ietf:params:netconf:capability:startup:1.0') }
              datastores << 'startup'
            end

            datastores
          end

          def generate_diff(before, after)
            # Convert the before/after configs to proper XML documents
            # Join XML strings and parse with REXML
            before_str = before.map(&:to_s).join
            after_str = after.map(&:to_s).join
            before_xml = REXML::Document.new(before_str)
            after_xml = REXML::Document.new(after_str)

            # Generate hierarchical diff with semantic understanding
            diff_result = generate_semantic_diff(before_xml.root, after_xml.root)

            {
              'summary' => diff_result['summary'],
              'changes' => diff_result['changes'],
              'before_state_captured' => true,
              'after_state_captured' => true
            }
          end

          # Generate a semantic diff that understands configuration structure
          def generate_semantic_diff(before_root, after_root)
            changes = []

            # Create maps of elements by their identifiers
            before_map = build_element_map(before_root)
            after_map = build_element_map(after_root)

            # Find removed elements (in before but not in after)
            before_map.each do |path, element|
              unless after_map[path]
                changes << create_removal_change(element, path)
              end
            end

            # Find added and modified elements
            after_map.each do |path, element|
              before_element = before_map[path]

              if before_element.nil?
                # This is an addition
                changes << create_addition_change(element, path)
              elsif elements_differ?(before_element, element)
                # This is a modification
                modification = create_modification_change(before_element, element, path)
                changes << modification if modification
              end
            end

            # Consolidate changes to avoid showing child additions when parent is added
            consolidated_changes = consolidate_changes(changes)

            {
              'changes' => consolidated_changes,
              'summary' => {
                'added' => consolidated_changes.count { |c| c['type'] == 'added' },
                'removed' => consolidated_changes.count { |c| c['type'] == 'removed' },
                'modified' => consolidated_changes.count { |c| c['type'] == 'modified' },
                'total' => consolidated_changes.length
              }
            }
          end

          # Build a map of all elements with their semantic paths
          def build_element_map(root, current_path = '', map = {})
            return map unless root

            # Process all child elements
            root.elements.each do |child|
              # Generate a semantic path for this element
              child_path = generate_semantic_path(child, current_path)

              # Only track meaningful configuration elements
              if meaningful_element?(child)
                map[child_path] = child
              end

              # Recurse into children
              build_element_map(child, child_path, map)
            end

            map
          end

          # Generate a human-readable semantic path for an element
          def generate_semantic_path(element, parent_path)
            # Identify the element type and its identifier
            identifier = find_element_identifier(element)

            if identifier
              # Use identifier for meaningful path
              "#{parent_path}/#{element.name}[#{identifier[:value]}]"
            else
              # For elements without identifiers, use the element name
              "#{parent_path}/#{element.name}"
            end
          end

          # Find the identifier for an element based on YANG key patterns
          def find_element_identifier(element)
            # Priority 1: Look for attributes that end with common identifier patterns
            # This handles vendor-specific attributes like 'vlan-id', 'interface-name', etc.
            identifier_patterns = %r{^(.+[-_])?(name|id|key|index|number|identifier)$}i

            element.attributes.each do |attr_name, attr|
              # Handle both regular attributes and namespace declarations
              attr_value = attr.respond_to?(:value) ? attr.value : attr
              if attr_name.match?(identifier_patterns) && !attr_value.to_s.strip.empty?
                return { type: attr_name, value: attr_value }
              end
            end

            # Priority 2: Look for child elements that are likely keys
            # In YANG, list keys are often direct children with simple text content
            element.elements.each do |child|
              # Skip if it has its own children (not a leaf)
              next unless child.elements.empty?

              # Check if child name suggests it's an identifier
              if child.name.match?(identifier_patterns) && !child.text.to_s.strip.empty?
                return { type: child.name, value: child.text.strip }
              end
            end

            # Priority 3: For YANG lists, the first leaf child is often the key
            # Only use this if there are multiple siblings with the same element name
            if element.parent
              siblings = element.parent.elements.select { |e| e.name == element.name }
              if siblings.length > 1
                # Find the first leaf child element
                first_leaf = element.elements.find do |child|
                  child.elements.empty? && !child.text.to_s.strip.empty?
                end

                if first_leaf
                  return { type: first_leaf.name, value: first_leaf.text.strip }
                end
              end
            end

            nil
          end

          # Check if an element represents meaningful configuration
          def meaningful_element?(element)
            # Skip YANG state containers (read-only data)
            return false if element.name == 'state'

            # Skip OpenConfig-style config containers that are just wrappers
            # But only if they have no attributes and their parent has the actual identifier
            if element.name == 'config' && element.attributes.empty?
              parent = element.parent
              if parent && find_element_identifier(parent)
                # This config element is just organizing data under an identified parent
                return false
              end
            end

            # Include all other elements - let the consolidation phase handle redundancy
            true
          end

          # Check if two elements differ in a meaningful way
          def elements_differ?(elem1, elem2)
            # Compare text content for leaf elements
            if elem1.elements.empty? && elem2.elements.empty?
              return elem1.text.to_s.strip != elem2.text.to_s.strip
            end

            # Compare attributes
            # Handle both regular attributes and namespace declarations
            # REXML stores namespace declarations as strings, not attribute objects
            attrs1 = elem1.attributes.map { |k, v| [k, v.respond_to?(:value) ? v.value : v] }.to_h
            attrs2 = elem2.attributes.map { |k, v| [k, v.respond_to?(:value) ? v.value : v] }.to_h
            return true if attrs1 != attrs2

            # For non-leaf elements, their children determine if they differ
            # This is handled by the recursive diff process
            false
          end

          # Create a removal change entry
          def create_removal_change(element, path)
            {
              'type' => 'removed',
              'path' => path,
              'description' => describe_element(element),
              'element_type' => element.name,
              'identifier' => find_element_identifier(element)
            }
          end

          # Create an addition change entry
          def create_addition_change(element, path)
            {
              'type' => 'added',
              'path' => path,
              'description' => describe_element(element),
              'element_type' => element.name,
              'identifier' => find_element_identifier(element),
              'details' => extract_element_details(element)
            }
          end

          # Create a modification change entry
          def create_modification_change(before_elem, after_elem, path)
            changes = {}

            # Check for text content changes (for leaf elements)
            if before_elem.elements.empty? && after_elem.elements.empty?
              before_text = before_elem.text.to_s.strip
              after_text = after_elem.text.to_s.strip

              if before_text != after_text
                changes['value'] = {
                  'old' => before_text,
                  'new' => after_text
                }
              end
            end

            # Check for attribute changes
            # Handle both regular attributes and namespace declarations
            before_attrs = before_elem.attributes.map { |k, v| [k, v.respond_to?(:value) ? v.value : v] }.to_h
            after_attrs = after_elem.attributes.map { |k, v| [k, v.respond_to?(:value) ? v.value : v] }.to_h

            if before_attrs != after_attrs
              changes['attributes'] = {
                'old' => before_attrs,
                'new' => after_attrs
              }
            end

            # Only return a change if something actually changed
            return nil if changes.empty?

            {
              'type' => 'modified',
              'path' => path,
              'description' => describe_element(after_elem),
              'element_type' => after_elem.name,
              'identifier' => find_element_identifier(after_elem),
              'changes' => changes
            }
          end

          # Generate a human-readable description of an element
          def describe_element(element)
            identifier = find_element_identifier(element)

            # Generic description based on element name and identifier
            element_display = element.name.split(%r{[-_]}).map(&:capitalize).join(' ')

            if identifier
              "#{element_display} '#{identifier[:value]}'"
            else
              element_display
            end
          end

          # Extract key details from an element for the change report
          def extract_element_details(element)
            details = {}

            # Extract meaningful child elements
            element.elements.each do |child|
              # Skip container elements
              next if ['config', 'state'].include?(child.name)

              if child.elements.empty? && child.text.to_s.strip != ''
                # Leaf element with text content
                details[child.name] = child.text.strip
              end
            end

            # Add attributes
            element.attributes.each do |name, attr|
              # Handle both regular attributes and namespace declarations
              details["@#{name}"] = attr.respond_to?(:value) ? attr.value : attr
            end

            details
          end

          # Consolidate changes to avoid redundancy
          def consolidate_changes(changes)
            # Group changes by their parent paths
            consolidated = []

            # Sort changes by path depth (fewer slashes = higher level)
            sorted_changes = changes.sort_by { |c| c['path'].count('/') }

            sorted_changes.each do |change|
              # Check if this change is already covered by a parent addition/removal
              parent_covered = false

              if ['added', 'removed'].include?(change['type'])
                consolidated.each do |existing|
                  next unless existing['type'] == change['type'] &&
                              change['path'].start_with?(existing['path'] + '/')
                  # This change is a child of an existing change
                  parent_covered = true
                  break
                end
              end

              consolidated << change unless parent_covered
            end

            consolidated
          end

          # Format changes for the final report
        end
      end
    end
  end
end
